package ru.yandex.client.pg;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.net.PemTrustOptions;
import io.vertx.pgclient.PgConnectOptions;
import io.vertx.pgclient.PgPool;
import io.vertx.pgclient.SslMode;
import io.vertx.sqlclient.PoolOptions;
import io.vertx.sqlclient.Row;
import io.vertx.sqlclient.RowSet;
import io.vertx.sqlclient.SqlConnection;
import io.vertx.sqlclient.Tuple;
import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.conn.routing.HttpRoute;

import ru.yandex.client.pg.config.ImmutablePgClientConfig;
import ru.yandex.concurrent.FailedFuture;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BasicFuture;
import ru.yandex.http.util.DuplexFutureCallback;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.client.RequestsListener;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;

public class PgClient implements GenericAutoCloseable<IOException> {
    private static final SqlQuery STATUS_QUERY =
        new SqlQuery("status-query.sql", PgClient.class);
    private static final Tuple EMPTY_TUPLE = Tuple.tuple();

    private final Timer timer = new Timer("PgPinger", true);
    private final Server[] servers;

    public PgClient(
        final Vertx vertx,
        final ImmutablePgClientConfig config,
        final PrefixedLogger logger)
        throws ConfigException, InterruptedException
    {
        PgConnectOptions connectOptions = new PgConnectOptions();
        connectOptions.setConnectTimeout(config.connectTimeout());
        connectOptions.setIdleTimeout(config.idleTimeout());
        connectOptions.setIdleTimeoutUnit(TimeUnit.MILLISECONDS);
        connectOptions.addProperty(
            "statement_timeout",
            Integer.toString(config.timeout()));

        connectOptions.setDatabase(config.database());
        connectOptions.setUser(config.user());
        connectOptions.setPassword(config.password());

        connectOptions.setCachePreparedStatements(
            config.preparedStatementsCacheSize() > 0);
        connectOptions.setPreparedStatementCacheMaxSize(
            config.preparedStatementsCacheSize());
        connectOptions.setSoLinger(1);
        File pemCertificate = config.pemCertificate();
        if (pemCertificate != null) {
            connectOptions.setSsl(true);
            connectOptions.setSslMode(SslMode.VERIFY_CA);
            connectOptions.setSslHandshakeTimeout(config.connectTimeout());
            connectOptions.setSslHandshakeTimeoutUnit(TimeUnit.MILLISECONDS);
            connectOptions.setTrustOptions(
                new PemTrustOptions()
                    .addCertPath(pemCertificate.getAbsolutePath()));
        }

        PoolOptions poolOptions = new PoolOptions();
        poolOptions.setConnectionTimeout(config.poolTimeout());
        poolOptions.setConnectionTimeoutUnit(TimeUnit.MILLISECONDS);
        poolOptions.setIdleTimeout(config.idleTimeout());
        poolOptions.setIdleTimeoutUnit(TimeUnit.MILLISECONDS);
        poolOptions.setMaxSize(config.connections());
        poolOptions.setMaxWaitQueueSize(config.queueSize());
        poolOptions.setPoolCleanerPeriod(1000);
        List<HttpHost> hosts = config.hosts();
        int size = hosts.size();
        servers = new Server[size];
        // TODO: wait only for first successful server instead of all of them
        CountDownLatch startLatch = new CountDownLatch(size);
        for (int i = 0; i < size; ++i) {
            HttpHost host = hosts.get(i);
            PgConnectOptions options = new PgConnectOptions(connectOptions);
            options.setHost(host.getHostName());
            options.setPort(host.getPort());
            Server server =
                new Server(host, PgPool.pool(vertx, options, poolOptions));
            servers[i] = server;
            timer.schedule(
                new InitServerTask(
                    timer,
                    server,
                    config.healthCheckInterval(),
                    logger.addPrefix(
                        host.getHostName() + ':' + host.getPort()),
                    startLatch), 0L);
        }
        startLatch.await();
        for (int i = 0; i < size; ++i) {
            if (servers[i].status().state() != Server.State.DEAD) {
                return;
            }
        }
        throw new ConfigException(
            "All servers " + hosts + " are unreachable");
    }

    @Override
    public void close() {
        timer.cancel();
        for (Server server: servers) {
            server.pool().close();
        }
    }

    public Future<RowSet<Row>> executeOnMaster(final SqlQuery query) {
        return executeOnMaster(
            query,
            EMPTY_TUPLE,
            EmptyFutureCallback.INSTANCE);
    }

    public Future<RowSet<Row>> executeOnMaster(
        final SqlQuery query,
        final Tuple queryParameters)
    {
        return executeOnMaster(
            query,
            queryParameters,
            EmptyFutureCallback.INSTANCE);
    }

    public Future<RowSet<Row>> executeOnMaster(
        final SqlQuery query,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        return executeOnMaster(query, EMPTY_TUPLE, callback);
    }

    public Future<RowSet<Row>> executeOnMaster(
        final SqlQuery query,
        final RequestsListener listener,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        return executeOnMaster(
            query,
            EMPTY_TUPLE,
            listener,
            callback);
    }

    public Future<RowSet<Row>> executeOnMaster(
        final SqlQuery query,
        final Tuple queryParameters,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        for (Server server: servers) {
            if (server.status().state() == Server.State.MASTER) {
                return execute(
                    query.query(),
                    queryParameters,
                    server.pool(),
                    callback);
            }
        }
        Exception e = new PgException(
            "No master found among " + Arrays.toString(servers));
        FailedFuture<RowSet<Row>> future = new FailedFuture<>(e);
        callback.failed(e);
        return future;
    }

    public Future<RowSet<Row>> executeOnAny(final SqlQuery query) {
        return executeOnAny(
            query,
            EMPTY_TUPLE,
            true,
            null,
            EmptyFutureCallback.INSTANCE);
    }

    public Future<RowSet<Row>> executeOnAny(
        final SqlQuery query,
        final Tuple queryParameters,
        final boolean shuffle,
        final RequestsListener listener,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        Server executer = null;
        if (shuffle) {
            List<Server> servers = new ArrayList<>(Arrays.asList(this.servers));
            Collections.shuffle(servers);
            for (Server server: servers) {
                if (server.status().state() != Server.State.DEAD) {
                    executer = server;
                    break;
                }
            }
        } else {
            for (Server server: servers) {
                if (server.status().state() != Server.State.DEAD) {
                    executer = server;
                    break;
                }
            }
        }

        if (executer != null) {
            FutureCallback<? super RowSet<Row>> resCallback = callback;
            if (listener != null) {
                resCallback = new DuplexFutureCallback<>(
                    listener.createCallbackFor(
                        new HttpRoute(
                            new HttpHost(query.queryName(), 0),
                            executer.host()),
                        query.queryName(),
                        RowSetResultInfo::new),
                    callback);
            }
            return execute(
                query.query(),
                queryParameters,
                executer.pool(),
                resCallback);
        }

        Exception e = new PgException(
            "No alive servers found among " + Arrays.toString(servers));
        FailedFuture<RowSet<Row>> future = new FailedFuture<>(e);
        callback.failed(e);
        return future;
    }

    public Future<RowSet<Row>> executeOnMaster(
        final SqlQuery query,
        final Tuple queryParameters,
        final RequestsListener listener,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        for (Server server: servers) {
            if (server.status().state() == Server.State.MASTER) {
                return execute(
                    query.query(),
                    queryParameters,
                    server.pool(),
                    new DuplexFutureCallback<>(
                        listener.createCallbackFor(
                            new HttpRoute(
                                new HttpHost(query.queryName(), 0),
                                server.host()),
                            query.queryName(),
                            RowSetResultInfo::new),
                        callback));
            }
        }
        Exception e = new PgException(
            "No master found among " + Arrays.toString(servers));
        FailedFuture<RowSet<Row>> future = new FailedFuture<>(e);
        callback.failed(e);
        return future;
    }

    public Future<RowSet<Row>> executeBatchOnMaster(
        final SqlQuery query,
        final List<Tuple> queriesParameters,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        for (Server server: servers) {
            if (server.status().state() == Server.State.MASTER) {
                return executeBatch(
                    query.query(),
                    queriesParameters,
                    server.pool(),
                    callback);
            }
        }
        Exception e = new PgException(
            "No master found among " + Arrays.toString(servers));
        FailedFuture<RowSet<Row>> future = new FailedFuture<>(e);
        callback.failed(e);
        return future;
    }

    public Future<RowSet<Row>> executeBatchOnMaster(
        final SqlQuery query,
        final List<Tuple> queriesParameters,
        final RequestsListener listener,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        for (Server server: servers) {
            if (server.status().state() == Server.State.MASTER) {
                return executeBatch(
                    query.query(),
                    queriesParameters,
                    server.pool(),
                    new DuplexFutureCallback<>(
                        listener.createCallbackFor(
                            new HttpRoute(
                                new HttpHost(query.queryName(), 0),
                                server.host()),
                            query.queryName(),
                            RowSetResultInfo::new),
                        callback));
            }
        }
        Exception e = new PgException(
            "No master found among " + Arrays.toString(servers));
        FailedFuture<RowSet<Row>> future = new FailedFuture<>(e);
        callback.failed(e);
        return future;
    }

    private static Future<RowSet<Row>> execute(
        final String query,
        final Tuple queryParameters,
        final PgPool pool,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        BasicFuture<RowSet<Row>> result = new BasicFuture<>(callback);
        pool.getConnection(
            new AsyncResultAdapter<>(
                new ConnectionCallback(result, query, queryParameters)));
        return result;
    }

    private static Future<RowSet<Row>> executeBatch(
        final String query,
        final List<Tuple> queriesParameters,
        final PgPool pool,
        final FutureCallback<? super RowSet<Row>> callback)
    {
        BasicFuture<RowSet<Row>> result = new BasicFuture<>(callback);
        pool.getConnection(
            new AsyncResultAdapter<>(
                new ConnectionBatchCallback(
                    result,
                    query,
                    queriesParameters)));
        return result;
    }

    private static class ConnectionCallback
        extends AbstractFilterFutureCallback<SqlConnection, RowSet<Row>>
    {
        private final String query;
        private final Tuple queryParameters;

        private ConnectionCallback(
            final FutureCallback<? super RowSet<Row>> callback,
            final String query,
            final Tuple queryParameters)
        {
            super(callback);
            this.query = query;
            this.queryParameters = queryParameters;
        }

        @Override
        public void completed(final SqlConnection connection) {
            Handler<AsyncResult<RowSet<Row>>> handler =
                new ConnectionClosingAsyncResultHandler<>(
                    new AsyncResultAdapter<>(callback),
                    connection);
            if (queryParameters.size() == 0) {
                connection.query(query).execute(handler);
            } else {
                connection.preparedQuery(query).execute(
                    queryParameters,
                    handler);
            }
        }
    }

    private static class ConnectionBatchCallback
        extends AbstractFilterFutureCallback<SqlConnection, RowSet<Row>>
    {
        private final String query;
        private final List<Tuple> queriesParameters;

        private ConnectionBatchCallback(
            final FutureCallback<? super RowSet<Row>> callback,
            final String query,
            final List<Tuple> queriesParameters)
        {
            super(callback);
            this.query = query;
            this.queriesParameters = queriesParameters;
        }

        @Override
        public void completed(final SqlConnection connection) {
            connection.preparedQuery(query).executeBatch(
                queriesParameters,
                new ConnectionClosingAsyncResultHandler<>(
                    new AsyncResultAdapter<>(callback),
                    connection));
        }
    }

    private static class ServerStatusTask
        extends TimerTask
        implements FutureCallback<RowSet<Row>>
    {
        private final Timer timer;
        private final Server server;
        private final long healthCheckInterval;
        private final PrefixedLogger logger;

        ServerStatusTask(
            final Timer timer,
            final Server server,
            final long healthCheckInterval,
            final PrefixedLogger logger)
        {
            this.timer = timer;
            this.server = server;
            this.healthCheckInterval = healthCheckInterval;
            this.logger = logger;
        }

        @Override
        public void run() {
            logger.info("Check availability for " + server);
            execute(
                STATUS_QUERY.query(),
                EMPTY_TUPLE,
                server.pool(),
                this);
        }

        private void reschedule() {
            try {
                timer.schedule(
                    new ServerStatusTask(
                        timer,
                        server,
                        healthCheckInterval,
                        logger),
                    healthCheckInterval);
            } catch (RuntimeException e) {
                logger.log(
                    Level.WARNING,
                    "Failed to reschedule task",
                    e);
            }
        }

        private void markDead() {
            server.status(new Server.Status(Server.State.DEAD, 0L));
        }

        @Override
        public void cancelled() {
            logger.warning("Service status task cancelled");
            markDead();
            reschedule();
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            int rows = rowSet.rowCount();
            boolean parseFailed = true;
            if (rows == 1) {
                try {
                    Row row = rowSet.iterator().next();
                    boolean master = row.getBoolean(0);
                    long lag = row.getLong(1);
                    Server.State state;
                    if (master) {
                        state = Server.State.MASTER;
                    } else {
                        state = Server.State.SLAVE;
                    }
                    Server.Status status = new Server.Status(state, lag);
                    logger.info("Server status: " + status);
                    server.status(status);
                    parseFailed = false;
                } catch (RuntimeException e) {
                    logger.log(
                        Level.WARNING,
                        "Failed to parse result",
                        e);
                }
            } else {
                logger.warning("Unexpected row count: " + rows);
            }
            if (parseFailed) {
                markDead();
            }
            reschedule();
        }

        @Override
        public void failed(final Exception e) {
            logger.log(
                Level.WARNING,
                "Status request failed",
                e);
            markDead();
            reschedule();
        }
    }

    private static class InitServerTask extends ServerStatusTask {
        private final CountDownLatch startLatch;

        InitServerTask(
            final Timer timer,
            final Server server,
            final long healthCheckInterval,
            final PrefixedLogger logger,
            final CountDownLatch startLatch)
        {
            super(timer, server, healthCheckInterval, logger);
            this.startLatch = startLatch;
        }

        @Override
        public void cancelled() {
            try {
                super.cancelled();
            } finally {
                startLatch.countDown();
            }
        }

        @Override
        public void completed(final RowSet<Row> rowSet) {
            try {
                super.completed(rowSet);
            } finally {
                startLatch.countDown();
            }
        }

        @Override
        public void failed(final Exception e) {
            try {
                super.failed(e);
            } finally {
                startLatch.countDown();
            }
        }
    }
}

