#include "common.h"

#include <library/cpp/json/json_writer.h>
#include <library/cpp/json/json_reader.h>

namespace NPyJson {

namespace py = boost::python;

py::object ToPyObject(const NJson::TJsonValue& v) {
    using namespace NJson;
    switch (v.GetType()) {
        case JSON_UNDEFINED: return {};
        case JSON_NULL: return {};
        case JSON_BOOLEAN: return py::object{v.GetBoolean()};
        case JSON_INTEGER: return py::object{v.GetInteger()};
        case JSON_UINTEGER: return py::object{v.GetUInteger()};
        case JSON_DOUBLE: return py::object{v.GetDouble()};
        case JSON_STRING: return py::str{v.GetString().data()};
        case JSON_MAP: {
                auto dict = py::dict();
                for (const auto& [key, val]: v.GetMap()) {
                    dict[key.c_str()] =  ToPyObject(val);
                }
                return dict;
            }
        case JSON_ARRAY: {
                auto list = py::list();
                for (const auto& item: v.GetArray()) {
                    list.append(ToPyObject(item));
                }
                return list;
            }
        // no default case specially to rise warning on compilation, to catch
        // changes in EJsonValueType enum with -Werror compiler options
    }
}

py::object ToPyObject(std::shared_ptr<NJson::TJsonValue> v) {
    return ToPyObject(*v);
}

NJson::TJsonValue FromPyObject(const py::object& obj) {
    using namespace NJson;
    py::extract<long> lng_(obj);
    py::extract<double> dbl_(obj);
    py::extract<const char*> str_(obj);
    py::extract<bool> bol_(obj);
    py::extract<py::dict> dct_(obj);
    py::extract<py::list> lst_(obj);
    return obj.is_none() ? TJsonValue{JSON_NULL} :
           // bol_.check() - false positive when obj is integer type
           //                2, for instance, is accepted as bool value true
           PyBool_Check(obj.ptr()) ? TJsonValue{bol_()} :
           lng_.check() ? TJsonValue{lng_()} :
           dbl_.check() ? TJsonValue{dbl_()} :
           str_.check() ? TJsonValue{str_()} :
           dct_.check() ? ({
                TJsonValue v(JSON_MAP);
                py::dict dict = dct_();
                const py::object items_gen = dict.items().attr("__iter__")();
                auto n = py::len(dict);
                for (decltype(n) idx = 0; idx < n; ++idx) {
#ifdef USE_PYTHON3
                    py::object item = items_gen.attr("__next__")();
#else
                    py::object item = items_gen.attr("next")();
#endif
                    py::object key = item[0];
                    py::object val = item[1];
                    py::extract<const char*> str(key);
                    if (str.check()) {
                        v[str()] = FromPyObject(val);
                    }
                }
                v;}) :
           lst_.check() ? ({
              TJsonValue v(JSON_ARRAY);
              py::list lst = lst_();
              auto n = py::len(lst);
              for (decltype(n) idx = 0; idx < n; ++idx) {
                  v.AppendValue(FromPyObject(lst[idx]));
              }
              v;}) :
           TJsonValue{TJsonValue::UNDEFINED};
}

} // namespace NPyJson

namespace {

using namespace NJson;
using namespace NPyJson;
namespace py = boost::python;

TJsonValue* FromObj(const py::object& obj) {
    return new TJsonValue{FromPyObject(obj)};
}

TJsonValue* FromObj(const TJsonValue& val) {
    return new TJsonValue{val};
}

std::shared_ptr<TJsonValue> CreateJsonValue() {
    return std::make_shared<TJsonValue>();
}

std::shared_ptr<TJsonValue> CreateJsonValue(const std::shared_ptr<TJsonValue>& val) {
    return std::shared_ptr<TJsonValue>(new TJsonValue{*val});
}

std::shared_ptr<TJsonValue> CreateJsonValue(const py::object& obj) {
    return std::shared_ptr<TJsonValue>(new TJsonValue{FromPyObject(obj)});
}

TPyJsonValue& DefConstructors(TPyJsonValue& c) {
    using namespace NJson;
    using namespace py;

    def("make_json_value",
        static_cast<TJsonValue*(*)(const py::object& obj)>(FromObj),
        return_value_policy<manage_new_object>());

    def("make_json_value",
        static_cast<TJsonValue*(*)(const TJsonValue& val)>(FromObj),
        return_value_policy<manage_new_object>());

    return c
        .def("__init__", make_constructor(static_cast<std::shared_ptr<TJsonValue>(*)()>(CreateJsonValue)))
        .def("__init__", make_constructor(static_cast<std::shared_ptr<TJsonValue>(*)(const object&)>(CreateJsonValue)))
        .def("__init__", make_constructor(static_cast<std::shared_ptr<TJsonValue>(*)(const std::shared_ptr<TJsonValue>& val)>(CreateJsonValue)))
        ;
}

TPyJsonValue& DefEquality(TPyJsonValue& c) {
    using namespace py;
    return c
        .def(py::self == py::self)
        .def(py::self != py::self);
}

BOOST_PYTHON_MODULE(json_value) {
    using namespace NJson;
    using namespace py;
    using namespace NPyJson;

    enum_<EJsonValueType>("EJsonValueType")
        .value("JSON_UNDEFINED", JSON_UNDEFINED)
        .value("JSON_NULL", JSON_NULL)
        .value("JSON_BOOLEAN", JSON_BOOLEAN)
        .value("JSON_INTEGER", JSON_INTEGER)
        .value("JSON_DOUBLE", JSON_DOUBLE)
        .value("JSON_STRING", JSON_STRING)
        .value("JSON_MAP", JSON_MAP)
        .value("JSON_ARRAY", JSON_ARRAY)
        .value("JSON_UINTEGER", JSON_UINTEGER)
        .export_values();

    auto c = TPyJsonValue("TJsonValue", no_init);

    DefConstructors(c);
    DefIndexableIntf(c);
    DefMapIntf(c);
    DefArrayIntf(c);
    DefTypeInfo(c);
    DefTypedView(c);
    DefStrRepr(c);
    DefExtra(c);
    DefEquality(c);
}

}
