#include "common.h"

namespace {

using namespace NJson;
using namespace NPyJson;
namespace py = boost::python;

TJsonValue* GetValueByPath(TJsonValue& self, const std::string& path, const std::string& delim) {
    if (delim.size() != 1) {
        PyErr(PyExc_ValueError, "Delimiter should be exactly 1 char.");
    }
    TJsonValue* result = self.GetValueByPath(path, delim[0]);
    if (result == nullptr) {
        PyErr(PyExc_ValueError, "Path does not exist in TJsonValue.");
    }
    return result;
}

void SetValueByPath(TJsonValue& self, const std::string& path, const TJsonValue& val, const std::string& delim) {
    if (delim.size() != 1) {
        PyErr(PyExc_ValueError, "Delimiter should be exactly 1 char.");
    }
    if (!self.SetValueByPath(path, val, delim[0])) {
        PyErr(PyExc_ValueError, "Path does not exist in TJsonValue.");
    }
}

void SetValueByPath(TJsonValue& self, const std::string& path, const py::object& val, const std::string& delim) {
    SetValueByPath(self, path, FromPyObject(val), delim);
}

void SetValue(TJsonValue& self, const py::object& val) {
    self.SetValue(FromPyObject(val));
}

void Scan(TJsonValue& self, const py::object& fn) {
    // fn(path, parent, value)
    // parent = None, value = self on top-level
    struct ScanFn : public IScanCallback {
        const py::object& fn;
        bool Do(const TString& path, TJsonValue*, TJsonValue& value) override {
            py::str p(path.data());
            py::object value_obj{std::make_shared<TJsonValue>(value)};
            py::object result_obj = fn(p, value_obj);
            py::extract<bool> result(result_obj);
            if (!PyBool_Check(result_obj.ptr())) {
                PyErr(PyExc_TypeError, "Callback function in TJsonValue.Scan method should return only True or False values.");
            }
            return result();
        }
        explicit ScanFn(const py::object& fn) : fn(fn) {}
    } scan{fn};
    self.Scan(scan);
}

} // namespace anonymous

namespace NPyJson {

TPyJsonValue& DefExtra(TPyJsonValue& c) {
    using namespace NJson;
    using namespace py;
    return c
        //SetValue
        .def("set_value_json_value",  static_cast<TJsonValue& (TJsonValue::*)(const TJsonValue&) >(&TJsonValue::SetValue), args("value"), return_self<>())
        .def("set_value", SetValue, args("value"), return_self<>())

        //Paths
        .def("get_value_by_path_json_value", GetValueByPath, (arg("path"), arg("delim")="."), return_internal_reference<>())
        .def("set_value_by_path_json_value",
             static_cast<void (*)(TJsonValue&, const std::string&, const TJsonValue&, const std::string&)>(SetValueByPath),
             (arg("path"), arg("value"), arg("delim")="."),
             return_self<>())

        .def("set_value_by_path",
             static_cast<void (*)(TJsonValue&, const std::string&, const object&, const std::string&)>(SetValueByPath),
             (arg("path"), arg("value"), arg("delim")="."),
             return_self<>())

        //Scan
        .def("scan", Scan, arg("visitor"), return_self<>());
}

} // namespace NPyJson
