#include "json_indexable_array.h"

namespace NPyJson::NIndexableArray{

using namespace NJson;
namespace py = boost::python;

namespace {

i64 NormalizeIdx(const TJsonValue& self, i64 idx) {
    auto sz = self.GetArray().size();
    if (idx < 0) {
        idx = SameTypeAs(sz + idx, idx);
    }
    if (idx < 0 || SameTypeAs(idx, sz) > sz ) {
        PyErr(PyExc_IndexError, "Index out of range");
    }
    return idx;
}

template<typename T>
bool NumbersEqual(const TJsonValue& jval, T val) {
    return (jval.IsInteger() && SameTypeAs(jval.GetIntegerRobust(), val) == val) ||
           (jval.IsUInteger() && SameTypeAs(jval.GetUIntegerRobust(), val) == val) ||
           (jval.IsDouble() && SameTypeAs(jval.GetDoubleRobust(), val) == val);
}

} // anonymous namespace

TJsonValue& GetItemArray(TJsonValue& self, const py::object& idx) {
    py::extract<size_t> intIdx(idx);
    if (intIdx.check()) {
        i64 i = intIdx();
        return self[NormalizeIdx(self, i)];
    } else {
        PyErr(PyExc_TypeError, "Wrong index type. expected Int");
    }
    return const_cast<TJsonValue&>(TJsonValue::UNDEFINED);
}

template<>
void SetItemArray<bool>(TJsonValue& self, const py::object& idx, bool val) {
    GetItemArray(self, idx) = val;
}

template<>
void SetItemArray<i64>(TJsonValue& self, const py::object& idx, i64 val) {
    GetItemArray(self, idx) = val;
}

template<>
void SetItemArray<ui64>(TJsonValue& self, const py::object& idx, ui64 val) {
    GetItemArray(self, idx) = val;
}

template<>
void SetItemArray<double>(TJsonValue& self, const py::object& idx, double val) {
    GetItemArray(self, idx) = val;
}

template<>
void SetItemArray<const char*>(TJsonValue& self, const py::object& idx, const char* val) {
    GetItemArray(self, idx) = val;
}

template<>
void SetItemArray<const py::list&>(TJsonValue& self, const py::object& idx, const py::list& val) {
    GetItemArray(self, idx) = FromPyObject(val);
}

template<>
void SetItemArray<const py::dict&>(TJsonValue& self, const py::object& idx, const py::dict& val) {
    GetItemArray(self, idx) = FromPyObject(val);
}

template<>
void SetItemArray<const py::object&>(TJsonValue& self, const py::object& idx, const py::object& val) {
    GetItemArray(self, idx) = FromPyObject(val);
}

template<>
void SetItemArray<const TJsonValue&>(TJsonValue& self, const py::object& idx, const TJsonValue& val) {
    GetItemArray(self, idx) = val;
}

void DelItemArray(TJsonValue& self, const py::object& idx) {
    py::extract<size_t> intIdx(idx);
    if (intIdx.check()) {
        i64 i = intIdx();
        self.EraseValue(NormalizeIdx(self, i));
    } else {
        PyErr(PyExc_TypeError, "Wrong index type. expected Int");
    }
}

template<typename T>
bool ContainsArray(const TJsonValue&, T);

template<>
bool ContainsArray<bool>(const TJsonValue& self, bool v) {
    for (const auto& item: self.GetArray()) {
        if (item.IsBoolean() && item.GetBooleanRobust() == v) {
            return true;
        }
    }
    return false;
}

template<>
bool ContainsArray<i64>(const TJsonValue& self, i64 v) {
    for (const auto& item: self.GetArray()) {
        if (NumbersEqual(item, v)) {
            return true;
        }
    }
    return false;
}

template<>
bool ContainsArray<ui64>(const TJsonValue& self, ui64 v) {
    for (const auto& item: self.GetArray()) {
        if (NumbersEqual(item, v)) {
            return true;
        }
    }
    return false;
}

template<>
bool ContainsArray<double>(const TJsonValue& self, double v) {
    for (const auto& item: self.GetArray()) {
        if (NumbersEqual(item, v)) {
            return true;
        }
    }
    return false;
}

template<>
bool ContainsArray<const char*>(const TJsonValue& self, const char* v) {
    for (const auto& item: self.GetArray()) {
        if (item.IsString() && item.GetStringRobust() == v) {
            return true;
        }
    }
    return false;
}

template<>
bool ContainsArray<const TJsonValue&>(const TJsonValue& self, const TJsonValue& v) {
    for (const auto& item: self.GetArray()) {
        if (item == v) {
            return true;
        }
    }
    return false;
}

template<>
bool ContainsArray<const py::list&>(const TJsonValue& self, const py::list& v) {
    auto val = FromPyObject(v);
    return ContainsArray<const TJsonValue&>(self, val);
}

template<>
bool ContainsArray<const py::dict&>(const TJsonValue& self, const py::dict& v) {
    auto val = FromPyObject(v);
    return ContainsArray<const TJsonValue&>(self, val);
}

} // anonymous namespace
