#include "common.h"

namespace {

using namespace NJson;
using namespace NPyJson;
namespace py = boost::python;

template<typename T>
void SetValue(TJsonValue& self, T val) {
    self = TJsonValue(val);
}

template<>
void SetValue<const py::list&>(TJsonValue& self, const py::list& lst) {
    self = FromPyObject(lst);
}

template<>
void SetValue<const py::dict&>(TJsonValue& self, const py::dict& dict) {
    self = FromPyObject(dict);
}

template<>
void SetValue<const py::object&>(TJsonValue& self, const py::object& obj) {
    self = FromPyObject(obj);
}

template<typename T>
T GetValue(TJsonValue&) { return {}; }

template<>
bool GetValue<bool>(TJsonValue& self) {
    if (!self.IsBoolean()) {
        PyErr(PyExc_TypeError, std::string{"JsonValue: expected JSON_BOOLEAN, got: "} + std::string{ToString(self.GetType())});
    }
    return self.GetBoolean();
}

template<>
const char* GetValue<const char*>(TJsonValue& self) {
    if (!self.IsString()) {
        PyErr(PyExc_TypeError, std::string{"JsonValue: expected JSON_STRING, got: "} + std::string{ToString(self.GetType())});
    }
    return self.GetString().data();
}

template<>
long long GetValue<long long>(TJsonValue& self) {
    if (!self.IsInteger()) {
        PyErr(PyExc_TypeError, std::string{"JsonValue: expected JSON_INTEGER, got: "} + std::string{ToString(self.GetType())});
    }
    return self.GetInteger();
}

template<>
size_t GetValue<size_t>(TJsonValue& self) {
    if (!self.IsUInteger()) {
        PyErr(PyExc_TypeError, std::string{"JsonValue: expected JSON_UINTEGER, got: "} + std::string{ToString(self.GetType())});
    }
    return self.GetUInteger();
}

template<>
double GetValue<double>(TJsonValue& self) {
    if (!self.IsDouble()) {
        PyErr(PyExc_TypeError, std::string{"JsonValue: expected JSON_DOUBLE, got: "} + std::string{ToString(self.GetType())});
    }
    return self.GetDouble();
}

template<>
py::list GetValue<py::list>(TJsonValue& self) {
    if (!self.IsArray()) {
        PyErr(PyExc_TypeError, std::string{"JsonValue: expected JSON_ARRAY, got: "} + std::string{ToString(self.GetType())});
    }
    return static_cast<py::list>(ToPyObject(self));
}

template<>
py::dict GetValue<py::dict>(TJsonValue& self) {
    if (!self.IsMap()) {
        PyErr(PyExc_TypeError, std::string{"JsonValue: expected JSON_MAP, got: "} + std::string{ToString(self.GetType())});
    }
    return static_cast<py::dict>(ToPyObject(self));
}

template<>
py::object GetValue<py::object>(TJsonValue& self) {
    return ToPyObject(self);
}

} // anonymous namespace

namespace NPyJson {

TPyJsonValue& DefTypeInfo(TPyJsonValue& c) {
    using namespace NJson;
    using namespace py;
    return c
        .def("get_type", &TJsonValue::GetType)
        .def("set_type", &TJsonValue::SetType, args("type"), return_self<>())
        .add_property("is_defined",  &TJsonValue::IsDefined)
        .add_property("is_null",     &TJsonValue::IsNull)
        .add_property("is_boolean",  &TJsonValue::IsBoolean)
        .add_property("is_double",   &TJsonValue::IsDouble)
        .add_property("is_string",   &TJsonValue::IsString)
        .add_property("is_map",      &TJsonValue::IsMap)
        .add_property("is_array",    &TJsonValue::IsArray)
        .add_property("is_integer",  &TJsonValue::IsInteger)
        .add_property("is_uinteger", &TJsonValue::IsUInteger);
}

TPyJsonValue& DefTypedView(TPyJsonValue& c) {
    using namespace NJson;
    using namespace py;
    return c
        .def("__int__", GetValue<long long>)
        .def("__bool__", GetValue<bool>)
        .add_property("as_boolean",  GetValue<bool>,        SetValue<bool>)
        .add_property("as_double",   GetValue<double>,      SetValue<double>)
        .add_property("as_integer",  GetValue<long long>,   SetValue<long long>)
        .add_property("as_uinteger", GetValue<size_t>,      SetValue<size_t>)
        .add_property("as_string",   GetValue<const char*>, SetValue<const char*>)
        .add_property("as_list",     GetValue<list>,        SetValue<const list&>)
        .add_property("as_dict",     GetValue<dict>,        SetValue<const dict&>)
        .add_property("as_object",   GetValue<object>,      SetValue<const object&>);
}

} // namespace NPyJson
