package ru.yandex.parser.query;

import java.io.IOException;
import java.io.StringReader;
import java.text.ParseException;
import java.util.ArrayDeque;
import java.util.Deque;

import ru.yandex.parser.query.generated.Tokenizer;

public class QueryParser {
    private final Deque<Token> tokens = new ArrayDeque<>();
    private final Deque<Integer> pars = new ArrayDeque<>();
    private final Tokenizer tokenizer;
    private Token token = null;

    public QueryParser(final String query) {
        this(new Tokenizer(new StringReader(query)));
    }

    public QueryParser(final Tokenizer tokenizer) {
        this.tokenizer = tokenizer;
    }

    private Token token() throws IOException, ParseException {
        token = tokens.poll();
        if (token == null) {
            token = tokenizer.nextToken().createToken(tokenizer);
        }
        return token;
    }

    private ParseException exc() {
        return new ParseException(
            "Unexpected token " + token + " at pos: " + tokenizer.pos(),
            tokenizer.pos());
    }

    private QueryAtom parseQuotedQuery() throws IOException, ParseException {
        tokenizer.state(Tokenizer.QUOTED_QUERY);
        QuotedQuery query = new QuotedQuery();
        while (true) {
            switch (token().type()) {
                case QUOTE:
                    tokenizer.unstate();
                    return query;
                case TOKEN:
                    query.tokens().add(new QueryToken(token));
                    break;
                case WHITESPACE:
                    break;
                default:
                    throw exc();
            }
        }
    }

    private QueryAtom parseFieldQuery() throws IOException, ParseException {
        tokenizer.state(Tokenizer.FIELD_QUERY);
        QueryAtom query;
        switch (token().type()) {
            case NOT:
                query = new NotQuery(parseFieldQuery());
                break;
            case LPAR:
                pars.push(tokenizer.pos());
                query = parse(Tokenizer.NESTED_FIELD_QUERY);
                break;
            case QUOTE:
                query = parseQuotedQuery();
                break;
            case TOKEN:
                query = new QueryToken(token);
                break;
            default:
                throw exc();
        }
        tokenizer.unstate();
        return query;
    }

    private QueryAtom tryParseAtom(final int nestedQueryState)
        throws IOException, ParseException
    {
        QueryAtom query;
        switch (token().type()) {
            case NOT:
                Token current = token;
                if (token().type() == Token.Type.WHITESPACE) {
                    query = new QueryToken(
                        new TextToken(Token.Type.TOKEN, "-", current.pos()));
                    tokens.push(token);
                } else {
                    tokens.push(token);
                    query = new NotQuery(parseAtom(nestedQueryState));
                }
                break;
            case LPAR:
                pars.push(tokenizer.pos());
                query = parse(nestedQueryState);
                break;
            case QUOTE:
                query = parseQuotedQuery();
                break;
            case FIELD:
                query = new FieldQuery(token, parseFieldQuery());
                break;
            case TOKEN:
                query = new QueryToken(token);
                break;
            default:
                tokens.push(token);
                query = null;
                break;
        }
        return query;
    }

    private QueryAtom parseAtom(final int nestedQueryState)
        throws IOException, ParseException
    {
        QueryAtom query = tryParseAtom(nestedQueryState);
        if (query == null) {
            throw exc();
        } else {
            return query;
        }
    }

    private QueryAtom parseOrAtom(final int nestedQueryState)
        throws IOException, ParseException
    {
        if (token().type() != Token.Type.WHITESPACE) {
            tokens.push(token);
        }
        QueryAtom query = parseAtom(nestedQueryState);
        while (true) {
            switch (token().type()) {
                case WHITESPACE:
                    QueryAtom subquery = tryParseAtom(nestedQueryState);
                    if (subquery == null) {
                        return query;
                    } else {
                        query = new AndQuery(query, subquery);
                        break;
                    }
                case AND:
                    query = new AndQuery(query, parseAtom(nestedQueryState));
                    break;
                default:
                    tokens.push(token);
                    return query;
            }
        }
    }

    private QueryAtom parse(final int nestedQueryState)
        throws IOException, ParseException
    {
        tokenizer.state(nestedQueryState);
        QueryAtom query = parseOrAtom(nestedQueryState);
        while (true) {
            switch (token().type()) {
                case OR:
                    query = new OrQuery(query, parseOrAtom(nestedQueryState));
                    break;
                case RPAR:
                    if (pars.isEmpty()) {
                        throw exc();
                    } else {
                        pars.pop();
                        tokenizer.unstate();
                        return query;
                    }
                case EOF:
                    if (pars.isEmpty()) {
                        tokenizer.unstate();
                        return query;
                    } else {
                        int pos = pars.peek();
                        throw new ParseException(
                            "Unclosed parenthesis at pos: " + pos,
                            pos);
                    }
                default:
                    throw exc();
            }
        }
    }

    public QueryAtom parse() throws IOException, ParseException {
        return parse(Tokenizer.YYINITIAL);
    }

    public static void main(final String... args) throws Exception {
        new QueryParser(args[0])
            .parse()
            .accept(new QueryPrintingVisitor(System.out));
    }
}

