package ru.yandex.parser.query;

import java.io.IOException;

public class QueryPrintingVisitor implements QueryVisitor<Void, IOException> {
    private final Appendable out;
    private int indent = 0;

    public QueryPrintingVisitor(final Appendable out) {
        this.out = out;
    }

    private void op(final String op) throws IOException {
        for (int i = 0; i < indent; ++i) {
            out.append('\t');
        }
        out.append(op);
        out.append('\n');
    }

    @Override
    public Void visit(final QueryAtom query) throws IOException {
        ++indent;
        query.accept(this);
        --indent;
        return null;
    }

    @Override
    public Void visit(final AndQuery query) throws IOException {
        visit(query.lhs());
        op("AND");
        visit(query.rhs());
        return null;
    }

    @Override
    public Void visit(final OrQuery query) throws IOException {
        visit(query.lhs());
        op("OR");
        visit(query.rhs());
        return null;
    }

    @Override
    public Void visit(final NotQuery query) throws IOException {
        op("NOT");
        visit(query.query());
        return null;
    }

    @Override
    public Void visit(final FieldQuery query) throws IOException {
        if (query.query() instanceof QueryToken) {
            op(
                query.fields().toString() + ':'
                + ((QueryToken) query.query()).text());
        } else {
            op(query.fields().toString() + ':');
            visit(query.query());
        }
        return null;
    }

    @Override
    public Void visit(final QuotedQuery query) throws IOException {
        op("" + '"');
        for (QueryToken token: query.tokens()) {
            visit(token);
        }
        op("" + '"');
        return null;
    }

    @Override
    public Void visit(final QueryToken query) throws IOException {
        op(query.text());
        return null;
    }
}

