#include "received_header.h"
#include "tls_utils.h"

#include <sstream>

namespace NReceived {

void AppendReceivedPartForSSL(std::ostringstream& stream, SSL* ssl) {
    if (!ssl) {
        return;
    }

    stream << "\r\n";

    const auto* cipher = SSL_get_current_cipher(ssl);
    int sslAlgbits = 0;
    int sslUsebits = SSL_CIPHER_get_bits(cipher, &sslAlgbits);

    stream << "\t(using " << SSL_get_version(ssl)
           << " with cipher " << SSL_CIPHER_get_name(cipher)
           << " (" << sslUsebits
           << "/" << sslAlgbits
           << " bits))\r\n";

    auto* peer = SSL_get_peer_certificate(ssl);
    if (peer) {
        std::string peerCN = GetTlsTextName(X509_get_subject_name(peer), NID_commonName);
        std::string issuerCN = GetTlsIssuerCN(peer);
        stream << "\t(Client CN \"" << peerCN << "\", Issuer \""
               << issuerCN << "\" ("
               << (SSL_get_verify_result(ssl) == X509_V_OK ?
                   "verified OK))" : "not verified))");
    } else {
        stream << "\t(Client certificate not present)";
    }
}

// https://tools.ietf.org/html/rfc822#section-4.1
std::string BuildReceivedHeaderValue(const TReceived& received) {
    if (received.Date.empty() ||
        received.LocalHost.empty() ||
        received.SessionId.empty() ||
        received.Protocol.empty())
    {
        throw std::logic_error("Empty date, local host, session id, protocol are not allowed");
    }

    std::ostringstream stream;
    if (!received.HeloHost.empty() && !received.RemoteHost.empty() && !received.RemoteIp.empty()) {
        stream << "from " << received.HeloHost << " (" << received.RemoteHost
               << " [" << received.RemoteIp << "])\r\n\t";
    }

    stream << "by " << received.LocalHost
           << (received.ClusterName.empty() ? "" : " (" + received.ClusterName + "/Yandex)")
           << " with " << received.Protocol << " id " << received.SessionId;

    if (!received.Recipient.empty()) {
        stream << "\r\n\tfor <" + received.Recipient + ">; " << received.Date;
    } else {
        stream << ";\r\n\t" << received.Date;
    }

    if (received.Ssl) {
        AppendReceivedPartForSSL(stream, received.Ssl);
    }
    return stream.str();
}

std::string BuildReceivedHeader(const TReceived& received) {
    return "Received: " + BuildReceivedHeaderValue(received) + "\r\n";
}

std::pair<std::string, std::string> BuildReceivedPair(const TReceived& received) {
    return {"Received", BuildReceivedHeaderValue(received)};
}

}  // namespace NReceived
