#include "tls_utils.h"

#include <ctype.h>

namespace NReceived {

std::size_t TrimZeroes(unsigned char* s, std::size_t l) {
    while ((l) > 0 && (s)[(l)-1] == 0) --(l);
    return l;
}

std::string GetTlsTextName(X509_NAME *name, int nid) {
    int pos = 0;
    X509_NAME_ENTRY* entry = nullptr;
    ASN1_STRING* entryStr = nullptr;
    int ch = 0;
    unsigned char* cp = nullptr;

    if (name == nullptr || (pos = X509_NAME_get_index_by_NID(name, nid, -1)) < 0) {
        return {};
    }
    if ((entry = X509_NAME_get_entry(name, pos)) == nullptr) {
        return {};
    }
    if ((entryStr = X509_NAME_ENTRY_get_data(entry)) == nullptr) {
        return {};
    }


    auto deleter = [](auto* ptr) {OPENSSL_free(ptr);};
    unsigned char* utf8Value = nullptr;
    std::unique_ptr<unsigned char, decltype(deleter)> guard{utf8Value, deleter};
    int utf8Length = ASN1_STRING_to_UTF8(&utf8Value, entryStr);

    if (utf8Length < 0) {
        return {};
    }

    utf8Length = TrimZeroes(utf8Value, utf8Length);
    if (utf8Length >= 256) {
        return {};
    }

    if (static_cast<std::size_t>(utf8Length) !=
        std::strlen(reinterpret_cast<const char*>(utf8Value)))
    {
        return {};
    }

    for (cp = utf8Value; (ch = *cp) != 0; cp++) {
        if (isascii(ch) && !isprint(ch)) {
            return {};
        }
    }

    return {utf8Value, utf8Value + utf8Length};
}

std::string GetTlsIssuerCN(X509 *peer) {
    X509_NAME *name = X509_get_issuer_name(peer);
    std::string cn = GetTlsTextName(name, NID_commonName);
    if (cn.empty()) {
        return GetTlsTextName(name, NID_organizationName);
    }
    return cn;
}

}  // namespace NReceived
