#include <mail/library/received_header/received_header.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/gtest.h>

#include <openssl/ssl.h>


using namespace NReceived;

const TReceived MINIMAL_RECEIVED{
    .Date = "Sun, 26 Apr 2020 16:28:37 +0300 (MSK)",
    .LocalHost = "localhost",
    .SessionId = "abcdefgh-abcdef",
};

TEST(BuildReceivedHeaderValue, SimpleCaseSuccess) {
    TReceived received = MINIMAL_RECEIVED;
    EXPECT_STREQ(
        "by localhost with SMTP id abcdefgh-abcdef;\r\n"
        "\tSun, 26 Apr 2020 16:28:37 +0300 (MSK)",
        BuildReceivedHeaderValue(received));
}

TEST(BuildReceivedHeaderValue, WithRemoteInfoSuccess) {
    TReceived received = MINIMAL_RECEIVED;

    received.HeloHost = "sas1-446aaa7d3c0d.qloud-c.yandex.net";
    received.RemoteHost = "localhost";
    received.RemoteIp = "127.0.0.1";

    EXPECT_STREQ(
        "from sas1-446aaa7d3c0d.qloud-c.yandex.net (localhost [127.0.0.1])\r\n"
        "\tby localhost with SMTP id abcdefgh-abcdef;\r\n"
        "\tSun, 26 Apr 2020 16:28:37 +0300 (MSK)",
        BuildReceivedHeaderValue(received));
}

TEST(BuildReceivedHeaderValue, WithRecipientSuccess) {
    TReceived received = MINIMAL_RECEIVED;

    received.Recipient = "foo@yandex.ru";

    EXPECT_STREQ(
        "by localhost with SMTP id abcdefgh-abcdef\r\n"
        "\tfor <foo@yandex.ru>; Sun, 26 Apr 2020 16:28:37 +0300 (MSK)",
        BuildReceivedHeaderValue(received));
}

TEST(BuildReceivedHeaderValue, WithProtocolSpecifiedSuccess) {
    TReceived received = MINIMAL_RECEIVED;

    received.Protocol = "LMTP";

    EXPECT_STREQ(
        "by localhost with LMTP id abcdefgh-abcdef;\r\n"
        "\tSun, 26 Apr 2020 16:28:37 +0300 (MSK)",
        BuildReceivedHeaderValue(received));
}

TEST(BuildReceivedHeaderValue, WithClusterNameSuccess) {
    TReceived received = MINIMAL_RECEIVED;

    received.ClusterName = "mxback";

    EXPECT_STREQ(
        "by localhost (mxback/Yandex) with SMTP id abcdefgh-abcdef;\r\n"
        "\tSun, 26 Apr 2020 16:28:37 +0300 (MSK)",
        BuildReceivedHeaderValue(received));
}

TEST(BuildReceivedHeaderValue, EmptyRequiredFieldsThrows) {
    TReceived received = MINIMAL_RECEIVED;

    EXPECT_NO_THROW(BuildReceivedHeaderValue(received));

    std::vector<TReceived> values(4, received);
    values[0].Date.clear();
    values[1].LocalHost.clear();
    values[2].SessionId.clear();
    values[3].Protocol.clear();

    for (auto&& value : values) {
        EXPECT_THROW(BuildReceivedHeaderValue(value), std::logic_error);
    }
}

TEST(BuildReceivedHeaderValue, WithSslSuccess) {
   TReceived received = MINIMAL_RECEIVED;

    auto ctxDeleter = [](auto* ctx) {SSL_CTX_free(ctx);};
    std::unique_ptr<SSL_CTX, decltype(ctxDeleter)> ctx( 
        SSL_CTX_new(SSLv23_server_method()),
        ctxDeleter
    );
    ASSERT_TRUE(ctx);

    auto sslDeleter = [](auto* ssl) {SSL_free(ssl);};
    std::unique_ptr<SSL, decltype(sslDeleter)> ssl(SSL_new(ctx.get()), sslDeleter);
    ASSERT_TRUE(ssl);

    received.Ssl = ssl.get();
    EXPECT_STREQ(
        "by localhost with SMTP id abcdefgh-abcdef;\r\n"
        "\tSun, 26 Apr 2020 16:28:37 +0300 (MSK)\r\n"
        "\t(using TLSv1.3 with cipher (NONE) (0/0 bits))\r\n"
        "\t(Client certificate not present)",
        BuildReceivedHeaderValue(received));
}

TEST(BuildReceivedHeader, SimpleCaseSuccess) {
    TReceived received = MINIMAL_RECEIVED;
    EXPECT_STREQ(
        "Received: by localhost with SMTP id abcdefgh-abcdef;\r\n"
        "\tSun, 26 Apr 2020 16:28:37 +0300 (MSK)\r\n",
        BuildReceivedHeader(received));
}
