package ru.yandex.cache.sqlite;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;

import ru.yandex.cache.CopyEntry;
import ru.yandex.cache.DBCache;
import ru.yandex.unsafe.NativeMemory2;
import ru.yandex.unsafe.NativeMemory2.NativeMemoryAllocator;
import ru.yandex.util.timesource.TimeSource;

public class SqliteCache2 implements DBCache {
    private static final NativeMemoryAllocator ALLOCATOR =
        NativeMemoryAllocator.get("SSDCacheTemp");
    private static final int MAX_ASYNC_TASKS = 50000;
    private static final long SIZE_CHECK_INTERVAL = 100;
    private static final long COUNT_CHECK_INTERVAL = 3000;
    private static final int DELETE_BATCH_SIZE = 1000;
    private static final int MAX_POOLED_DELAYED_BUFFERS = 100;
    private static final NativeMemory2 EMPTY_BUFFER = ALLOCATOR.alloc(2 + 2);
    private final Callable<Void> stop = new Callable<Void>() {
        @Override
        public Void call() {
            closeCache();
            return null;
        }
    };
    private final String dbPath;
    private SqlCacheUpdater updater;
    private final ThreadLocal<Long> cacheHandle = new ThreadLocal<>();
    private final ConcurrentLinkedQueue<NativeMemory2> delayedBuffers =
        new ConcurrentLinkedQueue<>();
    private final AtomicInteger delayedBuffersCount = new AtomicInteger(0);
    private final ConcurrentLinkedQueue<Long> handles =
        new ConcurrentLinkedQueue<>();
    private final LongAdder hits = new LongAdder();
    private final LongAdder misses = new LongAdder();
    private final boolean shared;

    static {
        System.loadLibrary("sqlite-cache");
        commonInit();
    }

    public SqliteCache2(final String dbPath, final long size) {
        this(dbPath, size, SIZE_CHECK_INTERVAL);
    }

    public SqliteCache2(
        final String dbPath,
        final long size,
        final long sizeCheckInterval)
    {
        if (dbPath.contains(":memory:") || dbPath.contains("mode=memory")) {
            this.dbPath = dbPath;
            shared = true;
        } else {
            shared = false;
            this.dbPath = dbPath + ".v2";
        }
        this.updater = new SqlCacheUpdater(size, sizeCheckInterval);
//        updater.start();
    }

    public void copyFrom(final SqliteCache sqlCache) throws IOException {
        for (CopyEntry entry = sqlCache.getOne();
            entry != null;
            entry = sqlCache.getOne())
        {
            put(
                entry.key(),
                entry.address(),
                entry.compressedSize(),
                entry.decompressedSize(),
                entry.accessFreq());
            sqlCache.remove(entry.key(), true);
            entry.close();
        }
    }

    @Override
    public long capacity() {
        return updater.maxSize();
    }

    @Override
    public long used() {
        return updater.currentSize();
    }

    @Override
    public long count() {
        return -1L;
//        return updater.keyCount();
    }

    @Override
    public void flush() {
        final Object lock = new Object();
        final AtomicBoolean finished = new AtomicBoolean(false);
        synchronized (lock) {
            updater.enqueueTask(
                new Callable<Void>() {
                    @Override
                    public Void call() throws IOException {
                        synchronized (lock) {
                            try {
                                updater.checkSize();
                            } finally {
                                finished.set(true);
                                lock.notifyAll();
                            }
                            System.err.println(
                                "Flushed, size="
                                + used());
                        }
                        return null;
                    }
                });
            while (!finished.get()) {
                try {
                    lock.wait();
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    private NativeMemory2 delayedBuffer(final int size) {
        NativeMemory2 mem = delayedBuffers.poll();
        if (mem == null) {
            delayedBuffersCount.incrementAndGet();
            return ALLOCATOR.alloc(size);
        } else {
            if (mem.size() < size) {
                mem.realloc(size);
            }
            return mem;
        }
    }

    //CSOFF: MagicNumber
    public String statsString() {
        long hits = this.hits.sum();
        long misses = this.misses.sum();
        double ratio =
            (hits * 100.0)
            / (double) (hits + misses);
        return "cacheHit: " + ratio
            + ", size=" + capacity()
            + ", currentSize=" + used();
    }
    //CSON: MagicNumber

    @Override
    public void close() {
        if (updater != null) {
            updater.enqueueTask(stop);
            try {
                updater.join();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            NativeMemory2 tmpBuf;
            while ((tmpBuf = delayedBuffers.poll()) != null) {
                tmpBuf.free();
            }
            updater = null;
        }
    }

    //CSOFF: ParameterNumber
    @Override
    public void put(
        final String key,
        final long address,
        final int len,
        final int decompressedLen,
        final int accessFreq,
        final boolean sync)
        throws IOException
    {
        if (sync) {
            put(key, address, len, decompressedLen, 1);
        } else {
            final NativeMemory2 tmpBuf;
            if (address != 0 && len > 0) {
                tmpBuf = delayedBuffer(len);
                NativeMemory2.unboxedCopy(
                    address,
                    tmpBuf.address(),
                    len);
            } else if (len == 0) {
                tmpBuf = EMPTY_BUFFER;
            } else {
                tmpBuf = null;
            }
            updater.enqueueTask(
                new Callable<Void>() {
                    @Override
                    @SuppressWarnings("ReferenceEquality")
                    public Void call() throws IOException {
                        try {
                            put(key, tmpBuf.address(), len, decompressedLen, 1);
                        } finally {
                            if (tmpBuf != null && tmpBuf != EMPTY_BUFFER) {
                                if (delayedBuffersCount.get()
                                    >= MAX_POOLED_DELAYED_BUFFERS)
                                {
                                    tmpBuf.free();
                                    delayedBuffersCount.decrementAndGet();
                                } else {
                                    delayedBuffers.add(tmpBuf);
                                }
                            }
                        }
                        return null;
                    }
                });
        }
    }

    private void put(
        final String key,
        final long address,
        final int len,
        final int decompressedLen,
        final int accessFreq)
        throws IOException
    {
        long handle = handle();
        int ret = put(handle, key, address, len, decompressedLen, accessFreq);
        if (ret == -1) {
            throw new IOException("cache put failed: "
                + lastError(handle));
        }
    }
    //CSON: ParameterNumber

    //CSOFF: ParameterNumber
    private static native int put(
        final long handle,
        final String key,
        final long valueAddress,
        final int len,
        final int decompressedLen,
        final int accessFreq);
    //CSON: ParameterNumber

    @Override
    public SqliteCacheEntry get(
        final String key,
        final boolean update)
        throws IOException
    {
        long handle = handle();
        SqliteCacheEntry entry = get(handle, key);
        if (entry == null) {
            throw new IOException("cache get failed: "
                + lastError(handle));
        }
        if (entry.compressedSize() == -1) {
            misses.add(1);
            return null;
        } else {
            hits.add(1);
        }
        if (update) {
            update(key, false);
        }
        return entry;
    }

    private static native SqliteCacheEntry get(
        final long handle,
        final String key);

    @Override
    public SqliteCopyEntry getOne() throws IOException {
        long handle = handle();
        SqliteCopyEntry entry = getOne(handle);
        if (entry == null) {
            throw new IOException("cache getOne failed: "
                + lastError(handle));
        }
        if (entry.compressedSize() == -1) {
            return null;
        }
        return entry;
    }

    private static native SqliteCopyEntry getOne(final long handle);

    @Override
    public void update(final String key, final boolean sync)
        throws IOException
    {
        if (sync) {
            update(key);
        } else {
            updater.enqueueTask(
                new Callable<Void>() {
                    @Override
                    public Void call() throws IOException {
                        update(key);
                        return null;
                    }
                });
        }
    }

    public void update(final String key) throws IOException {
        long handle = handle();
        int ret = update(handle, key);
        if (ret == -1) {
            throw new IOException("cache update failed: "
                + lastError(handle));
        }
    }

    private static native int update(final long handle, final String key);

    @Override
    public void remove(final String key, final boolean sync)
        throws IOException
    {
        if (sync) {
            remove(key);
        } else {
            updater.enqueueTask(
                new Callable<Void>() {
                    @Override
                    public Void call() throws IOException {
                        remove(key);
                        return null;
                    }
                });
        }
    }

    public void remove(final String key) throws IOException {
        long handle = handle();
        int ret = remove(handle, key);
        if (ret == -1) {
            throw new IOException("cache remove failed: "
                + lastError(handle));
        }
    }

    private static native int remove(final long handle, final String key);

    @Override
    public void removePrefix(final String key, final boolean sync)
        throws IOException
    {
        if (sync) {
            removePrefix(key);
        } else {
            updater.enqueueTask(
                new Callable<Void>() {
                    @Override
                    public Void call() throws IOException {
                        removePrefix(key);
                        return null;
                    }
                });
        }
    }

    public void removePrefix(final String key) throws IOException {
        long handle = handle();
        int ret = removePrefix(handle, key + '%');
        if (ret == -1) {
            throw new IOException("cache removePrefix failed: "
                + lastError(handle));
        }
    }

    private static native int removePrefix(
        final long handle,
        final String prefix);

    @Override
    public void deleteTop(final int count) throws IOException {
        long handle = handle();
        int ret = deleteTop(handle, count);
        if (ret == -1) {
            throw new IOException("delete top failed: "
                + lastError(handle));
        }
    }

    private static native int deleteTop(final long handle, final int count);

    @Override
    public long cacheSize() throws IOException {
        long handle = handle();
        long ret = cacheSize(handle);
        if (ret == -1) {
            throw new IOException("cacheSize error: "
                + lastError(handle));
        }
        return ret;
    }

    private static native long cacheSize(final long handle);

    private long handle() throws IOException {
        Long handle = cacheHandle.get();
        if (handle == null) {
            handle = init(dbPath, shared);
            if (handle == 0) {
                throw new IOException(
                    "Can't initialize cache handle: "
                    + lastError(handle));
            }
            cacheHandle.set(handle);
            handles.add(handle);
        }
        return handle;
    }

    private void closeCache() {
        Long handle;
        do {
            handle = handles.poll();
            if (handle != null) {
                closeHandle(handle);
            }
        } while (handle != null);
    }

    private static native long init(final String dbPath, boolean shared)
        throws IOException;

    private static native void closeHandle(final long handle);

//    private static native long copyBlob(final long blob, final int len);

    private static native String lastError(final long handle);

    private static native void commonInit();

    public static native void freeBlob(final long blob);

    public static native int keyCount(final long handle);

    public static native int crc32(final long blob, final int len);

    @Override
    public String[] uniqFiles() throws IOException {
        long handle = handle();
        String[] ret = uniqFiles(handle);
        if (ret == null) {
            throw new IOException("uniqFiles error: "
                + lastError(handle));
        }
        return ret;
    }

    public static native String[] uniqFiles(final long handle);

    private final class SqlCacheUpdater extends Thread {
        private final long sizeCheckInterval;
        private final long size;
        private final long sizeHiThreshold;
        private final long sizeLoThreshold;
        private ArrayBlockingQueue<Callable<Void>> queue =
            new ArrayBlockingQueue<>(MAX_ASYNC_TASKS);
        private volatile long currentSize;
//        private volatile long keyCount;

        SqlCacheUpdater(final long size, final long sizeCheckInterval) {
            super("Sql2CacheUpdater");
            this.size = size;
            this.sizeCheckInterval = sizeCheckInterval;
            long delta = size >> (2 + 2 + 2);
            sizeHiThreshold = size - delta;
            sizeLoThreshold = size - delta - delta;
            setDaemon(true);
            start();
        }

        public void enqueueTask(final Callable<Void> task) {
            try {
                queue.put(task);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }

        public long currentSize() {
            return currentSize;
        }

        public long maxSize() {
            return size;
        }

//        public long keyCount() {
//            return keyCount;
//        }

        public void checkSize() throws IOException {
            currentSize = cacheSize();
            if (currentSize > sizeHiThreshold) {
                System.err.println(
                    "sqlite cache size overlimit: " + currentSize);
                while (currentSize > sizeLoThreshold) {
                    System.err.println(
                        "size: " + currentSize + ", Deleting top");
                    deleteTop(DELETE_BATCH_SIZE);
                    currentSize = cacheSize();
                }
                System.err.println(
                    "sqlite cache trimmed: " + currentSize);
            }
        }

        @Override
        @SuppressWarnings("CatchAndPrintStackTrace")
        public void run() {
            long prevSizeCheck = TimeSource.INSTANCE.currentTimeMillis();
            long prevCountCheck = prevSizeCheck;
            while (true) {
                try {
                    Callable<Void> task =
                        queue.poll(SIZE_CHECK_INTERVAL, TimeUnit.MILLISECONDS);
                    if (task != null) {
                        task.call();
                    }
                    if (task == stop) {
                        break;
                    }
                    long time = TimeSource.INSTANCE.currentTimeMillis();
                    if (time - prevSizeCheck > sizeCheckInterval) {
                        prevSizeCheck = time;
                        checkSize();
                    }
                    if (time - prevCountCheck > COUNT_CHECK_INTERVAL) {
                        prevCountCheck = time;
//                        keyCount = SqliteCache2.keyCount(handle());
                    }
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
    }
}
