#define _XOPEN_SOURCE 100500

#include <contrib/libs/sqlite3/sqlite3.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <jni.h>

#define CREATE_TABLE_STMT "BEGIN; \
        CREATE TABLE IF NOT EXISTS cache \
        (key TEXT PRIMARY KEY, \
        data BLOB, \
        decompressedSize INTEGER, \
        lastAccess INTEGER DEFAULT CURRENT_TIMESTAMP, \
        accessFreq INTEGER DEFAULT 1); \
        CREATE INDEX IF NOT EXISTS idx ON cache (accessFreq, lastAccess); \
        COMMIT;"

typedef struct {
    sqlite3_stmt *putStatement;
    sqlite3_stmt *updateStatement;
    sqlite3_stmt *getStatement;
    sqlite3_stmt *removeStatement;
    sqlite3_stmt *removePrefixStatement;
    sqlite3_stmt *countPrefixStatement;
    sqlite3_stmt *pragmaPageCountStatement;
    sqlite3_stmt *pragmaPageSizeStatement;
    sqlite3_stmt *pragmaFreePagesStatement;
    sqlite3_stmt *deleteTopStatement;
    sqlite3_stmt *getOneStatement;
    sqlite3_stmt *uniqFilesStatement;
    sqlite3 *db;
    char *lastError;
    int lastErrorBufferSize;
} CacheHandle_t;

void setLastError(
    CacheHandle_t *handle,
    const char *method,
    int code)
{
    const char *errMsg;
    if (handle->db != NULL) {
        errMsg = sqlite3_errmsg(handle->db);
    } else {
        errMsg = "unknown error";
    }
    int expectedSize =
        snprintf(
            NULL,
            0,
            "%s SQL error: code=%d: %s",
            method,
            code,
            errMsg) + 1;
    if (handle->lastErrorBufferSize < expectedSize) {
        handle->lastErrorBufferSize = expectedSize << 1;
        handle->lastError =
            (char *) realloc(handle->lastError, handle->lastErrorBufferSize);
    }
    snprintf(
        handle->lastError,
        handle->lastErrorBufferSize,
        "%s SQL error: code=%d: %s",
        method,
        code,
        errMsg);
    fprintf(stderr, "%s\n", handle->lastError);
}

int commonInit(CacheHandle_t* ch) {
    (void) ch;
/*
    int rc = sqlite3_config(SQLITE_CONFIG_MULTITHREAD);
    if (rc) {
        setLastError(
            ch,
            "commonInit.enableMultiThreadMode",
            rc);
        return -1;
    }

    rc = sqlite3_enable_shared_cache(0);
    if (rc) {
        setLastError(
            ch,
            "commonInit.enadleSharedCache", rc);
        return -1;
    }
    return 0;
*/
    return 0;
}

int openDatabase(
    CacheHandle_t *ch,
    const char *name,
    jboolean shared)
{
    int flags = 
        SQLITE_OPEN_NOMUTEX
            | SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE;
    if (shared) {
        flags |= SQLITE_OPEN_URI;
//        flags |= SQLITE_OPEN_SHAREDCACHE | SQLITE_OPEN_URI;
    } else {
        flags |= SQLITE_OPEN_PRIVATECACHE;
    }
    int rc = sqlite3_open_v2(
        name,
        &ch->db,
        flags,
        NULL);
    if (rc) {
        setLastError(
            ch,
            "openDatabase.open",
            rc);
        sqlite3_close(ch->db);
        return -1;
    }
    char* zErrMsg = 0;
    sqlite3_exec(ch->db, "PRAGMA synchronous = OFF", NULL, NULL, &zErrMsg);
    if (rc) {
        setLastError(
            ch,
            "openDatabase.pragma_synchronous",
            rc);
        sqlite3_free(zErrMsg);
        sqlite3_close(ch->db);
        return -1;
    }
    /*sqlite3_exec(db, "PRAGMA journal_mode = OFF", NULL, NULL, &zErrMsg);*/
    sqlite3_exec(ch->db, "PRAGMA journal_mode = WAL", NULL, NULL, &zErrMsg);
    if (rc) {
        setLastError(
            ch,
            "openDatabase.pragma_journal_mode",
            rc);
        sqlite3_free(zErrMsg);
        sqlite3_close(ch->db);
        return -1;
    }
    sqlite3_exec(ch->db, "PRAGMA busy_timeout = 30000", NULL, NULL, &zErrMsg);
    if (rc) {
        setLastError(
            ch,
            "openDatabase.pragma_busy_timeout",
            rc);
        sqlite3_free(zErrMsg);
        sqlite3_close(ch->db);
        return -1;
    }
    sqlite3_exec(ch->db, "PRAGMA cache_size = -100000", NULL, NULL, &zErrMsg);
    if (rc) {
        setLastError(
            ch,
            "openDatabase.pragma_cache_size",
            rc);
        sqlite3_free(zErrMsg);
        sqlite3_close(ch->db);
        return -1;
    }
    sqlite3_exec(
        ch->db,
        "PRAGMA journal_size_limit = 134217728",
        NULL,
        NULL,
        &zErrMsg);
    if (rc) {
        setLastError(
            ch,
            "openDatabase.journal_size_limit",
            rc);
        sqlite3_free(zErrMsg);
        sqlite3_close(ch->db);
        return -1;
    }
    sqlite3_exec(
        ch->db,
        "PRAGMA read_uncommitted = ON",
        NULL,
        NULL,
        &zErrMsg);
    if (rc) {
        setLastError(
            ch,
            "openDatabase.read_uncommited",
            rc);
        sqlite3_free(zErrMsg);
        sqlite3_close(ch->db);
        return -1;
    }

    sqlite3_exec(
        ch->db,
        "PRAGMA case_sensitive_like = ON",
        NULL,
        NULL,
        &zErrMsg);
    if (rc) {
        setLastError(
            ch,
            "openDatabase.case_sensitive_like",
            rc);
        sqlite3_free(zErrMsg);
        sqlite3_close(ch->db);
        return -1;
    }

    return 0;
}

int createTableCallback(void *unused, int argc, char **argv, char **azColName) {
    (void) unused;
    int i;
    for(i=0; i<argc; i++) {
        printf("%s = %s\n", azColName[i], argv[i] ? argv[i] : "NULL");
    }
    printf("\n");
    return 0;
}

int createTables(CacheHandle_t* ch) {
    char* zErrMsg = 0;
    int rc = sqlite3_exec(
        ch->db,
        CREATE_TABLE_STMT,
        createTableCallback,
        0,
        &zErrMsg);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "createTables",
            rc);
        sqlite3_free(zErrMsg);
        return 1;
    }
    return 0;
}

int prepareStatements(CacheHandle_t* ch) {
    const char* pzTail = 0;
    int rc = sqlite3_prepare_v2(
        ch->db,
        "INSERT INTO cache(key, data, decompressedSize, accessFreq) VALUES(?, ?, ?, ?) "
            "ON CONFLICT(key) "
            "DO UPDATE SET lastAccess=CURRENT_TIMESTAMP, "
            "accessFreq=accessFreq+1",
        -1,
        &ch->putStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.putStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "UPDATE cache SET "
            "lastAccess=julianday(\"now\"), accessFreq=accessFreq+1 "
            "WHERE key=?",
        -1,
        &ch->updateStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.updateStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "SELECT data,decompressedSize FROM cache WHERE key=?",
        -1,
        &ch->getStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.getStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
//        "SELECT data,decompressedSize FROM cache WHERE key=?",
        "SELECT data,decompressedSize,accessFreq,key FROM cache LIMIT 1",
        -1,
        &ch->getOneStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.getOneStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "SELECT DISTINCT substr("
            "rtrim(key,'0123456789'),"
            "1,"
            "length(rtrim(key,'0123456789')) - 1) from cache",
        -1,
        &ch->uniqFilesStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.uniqFilesStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "DELETE FROM cache WHERE key=?",
        -1,
        &ch->removeStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.removeStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
//        "DELETE FROM cache WHERE key in "
//            "(SELECT key FROM cache WHERE key like ?)",
        "DELETE FROM cache WHERE key in "
            "(SELECT key FROM cache WHERE key like ? limit ?)",
//        "DELETE FROM cache WHERE key like ?",
        -1,
        &ch->removePrefixStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.removePrefixStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "SELECT count(key) from cache where key like ?",
        -1,
        &ch->countPrefixStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.countPrefixStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "PRAGMA freelist_count",
        -1,
        &ch->pragmaFreePagesStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.pragmaFreePagesStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "PRAGMA page_count",
        -1,
        &ch->pragmaPageCountStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.pragmaPageCountStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "PRAGMA page_size",
        -1,
        &ch->pragmaPageSizeStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.pragmaPageSizeStatement",
            rc);
        return 1;
    }

    rc = sqlite3_prepare_v2(
        ch->db,
        "delete from cache where key in "
            "(select key from cache order by accessFreq, lastAccess limit ?)",
        -1,
        &ch->deleteTopStatement,
        &pzTail);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "prepareStatements.deleteTopStatement",
            rc);
        return 1;
    }

    return 0;
}

int cachePut(
    CacheHandle_t *ch,
    const char *key,
    const void *data,
    int len,
    int decompressedSize,
    int accessFreq)
{
//    for (int i = 0; i < 100; i++) {
//        fprintf(stderr, "put %x", ((char*)data)[i]);
//    }
//    fprintf(stderr, "\n");
    int ret = -1;
    int rc = sqlite3_bind_text(
        ch->putStatement,
        1,
        key,
        -1,
        0);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "cachePut.bind_text_1",
            rc);
        goto exit;
    }
    rc = sqlite3_bind_blob(
        ch->putStatement,
        2,
        data,
        len,
        0);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "cachePut.bind_blob_2",
            rc);
        goto exit;
    }
    rc = sqlite3_bind_int(
        ch->putStatement,
        3,
        decompressedSize);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "cachePut.bind_int_3",
            rc);
        goto exit;
    }
    rc = sqlite3_bind_int(
        ch->putStatement,
        4,
        accessFreq);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "cachePut.bind_int_4",
            rc);
        goto exit;
    }

    rc = sqlite3_step(ch->putStatement);
    if (rc != SQLITE_DONE) {
        setLastError(
            ch,
            "cachePut.step",
            rc);
        goto exit;
    }
    ret = 0;

exit:
    sqlite3_reset(ch->putStatement);

    return ret;
}

int cacheUpdate(
    CacheHandle_t *ch,
    const char *key)
{
    int ret = -1;
    int rc = sqlite3_bind_text(
        ch->updateStatement,
        1,
        key,
        -1,
        0);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "cacheUpdate.bind_text_1",
            rc);
        goto exit;
    }

    rc = sqlite3_step(ch->updateStatement);
    if (rc != SQLITE_DONE) {
        setLastError(
            ch,
            "cacheUpdate.step",
            rc);
        goto exit;
    }
    ret = 0;

exit:
    sqlite3_reset(ch->updateStatement);

    return ret;
}

int cacheRemove(CacheHandle_t* ch, const char *key) {
    int ret = -1;
    int rc = sqlite3_bind_text(
        ch->removeStatement,
        1,
        key,
        -1,
        0);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "cacheRemove.bind_text",
            rc);
        goto error;
    }
    rc = sqlite3_step(ch->removeStatement);
    if (rc != SQLITE_DONE) {
        setLastError(
            ch,
            "cacheRemove.step",
            rc);
        goto error;
    }

    ret = 0;

error:
    sqlite3_reset(ch->removeStatement);

    return ret;
}

int cacheRemovePrefix1(CacheHandle_t* ch, const char *key) {
    int ret = -1;
        int rc = sqlite3_bind_text(
            ch->removePrefixStatement,
            1,
            key,
            -1,
            0);
        if (rc != SQLITE_OK) {
            setLastError(
                ch,
                "cacheRemovePrefix.bind_text",
                rc);
            sqlite3_reset(ch->removePrefixStatement);
            goto exit;
        }
/*
        rc = sqlite3_bind_int(
            ch->removePrefixStatement,
            2,
            1000000000);
        if (rc != SQLITE_OK) {
            setLastError(
                ch,
                "cacheRemovePrefix.bind_int",
                rc);
            sqlite3_reset(ch->removePrefixStatement);
            goto exit;
        }
*/
        rc = sqlite3_step(ch->removePrefixStatement);
        if (rc != SQLITE_DONE) {
            setLastError(
                ch,
                "cacheRemovePrefix.step",
                rc);
            sqlite3_reset(ch->removePrefixStatement);
            goto exit;
        }
        sqlite3_reset(ch->removePrefixStatement);

    ret = 0;

exit:
    return ret;
}

int countPrefix(CacheHandle_t* ch, const char *key) {
    int ret = -1;
    int rc = sqlite3_bind_text(
        ch->countPrefixStatement,
        1,
        key,
        -1,
        0);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "cacheRemovePrefix.countPrefixStatement.bind_text",
            rc);
        sqlite3_reset(ch->countPrefixStatement);
        goto exit;
    }
    rc = sqlite3_step(ch->countPrefixStatement);
    if (rc != SQLITE_ROW) {
        setLastError(
            ch,
            "cacheSize.step.countPrefixStatement",
            rc);
        sqlite3_reset(ch->countPrefixStatement);
        goto exit;
    }
    long prefixKeyCount = sqlite3_column_int64(
        ch->countPrefixStatement,
        0);
    sqlite3_reset(ch->countPrefixStatement);
    ret = prefixKeyCount;
exit:
    return ret;
}

int cacheRemovePrefix(CacheHandle_t* ch, const char *key) {
    int ret = -1;
    while (1) {
        long prefixKeyCount = countPrefix(ch, key);
        if (prefixKeyCount == -1) {
            goto exit;
        } else if (prefixKeyCount == 0) {
            break;
        }
        while (prefixKeyCount > 0) {
            int rc = sqlite3_bind_text(
                ch->removePrefixStatement,
                1,
                key,
                -1,
                0);
            if (rc != SQLITE_OK) {
                setLastError(
                    ch,
                    "cacheRemovePrefix.bind_text",
                    rc);
                sqlite3_reset(ch->removePrefixStatement);
                goto exit;
            }
            rc = sqlite3_bind_int(
                ch->removePrefixStatement,
                2,
                500);
            if (rc != SQLITE_OK) {
                setLastError(
                    ch,
                    "cacheRemovePrefix.bind_int",
                    rc);
                sqlite3_reset(ch->removePrefixStatement);
                goto exit;
            }

            rc = sqlite3_step(ch->removePrefixStatement);
            if (rc != SQLITE_DONE) {
                setLastError(
                    ch,
                    "cacheRemovePrefix.step",
                    rc);
                sqlite3_reset(ch->removePrefixStatement);
                goto exit;
            }
            sqlite3_reset(ch->removePrefixStatement);
            prefixKeyCount -= 500;
        }
    }

    ret = 0;

exit:
    return ret;
}

int deleteTop(CacheHandle_t* ch, int count) {
    int ret = -1;
    int rc = sqlite3_bind_int(
        ch->deleteTopStatement,
        1,
        count);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "deleteTop.bind_int",
            rc);
        goto exit;
    }
    rc = sqlite3_step(ch->deleteTopStatement);
    if (rc != SQLITE_DONE) {
        setLastError(
            ch,
            "deleteTop.step",
            rc);
        goto exit;
    }
    ret = 0;

exit:
    sqlite3_reset(ch->deleteTopStatement);

    return ret;
}

long cacheSize(CacheHandle_t* ch) {
    int rc = sqlite3_step(ch->pragmaFreePagesStatement);
    long ret = -1;
    if (rc != SQLITE_ROW) {
        setLastError(
            ch,
            "cacheSize.step.freePages",
            rc);
        sqlite3_reset(ch->pragmaFreePagesStatement);
        goto exit;
    }
    long freePages = sqlite3_column_int(ch->pragmaFreePagesStatement, 0);
    sqlite3_reset(ch->pragmaFreePagesStatement);

    rc = sqlite3_step(ch->pragmaPageCountStatement);
    if (rc != SQLITE_ROW) {
        setLastError(
            ch,
            "cacheSize.step.pageCount",
            rc);
        sqlite3_reset(ch->pragmaPageCountStatement);
        goto exit;
    }
    long pageCount = sqlite3_column_int64(ch->pragmaPageCountStatement, 0);
    sqlite3_reset(ch->pragmaPageCountStatement);

    rc = sqlite3_step(ch->pragmaPageSizeStatement);
    if (rc != SQLITE_ROW) {
        setLastError(
            ch,
            "cacheSize.step.pageSize",
            rc);
        sqlite3_reset(ch->pragmaPageSizeStatement);
        goto exit;
    }
    long pageSize = sqlite3_column_int64(ch->pragmaPageSizeStatement, 0);
    sqlite3_reset(ch->pragmaPageSizeStatement);

    ret = (pageCount - freePages) * pageSize;

exit:

    return ret;
}

char** cacheUniqFiles(
    CacheHandle_t* ch,
    int *count)
{
    char** ret;
    int maxCount = 32;
    int i;
    ret = (char**) malloc(sizeof(char*) * maxCount);
    *count = 0;
    while (1) {
        int rc = sqlite3_step(ch->uniqFilesStatement);
        if (rc != SQLITE_ROW) {
            if (rc == SQLITE_DONE) {
                goto done;
            }
            setLastError(
                ch,
                "cacheUniqFiles.step error",
                rc);
            goto error;
        }
        const char *file =
            (const char*) sqlite3_column_text(ch->uniqFilesStatement, 0);
        ret[count[0]++] = strdup((const char *)file);
        if (*count == maxCount) {
            maxCount <<= 1;
            ret = (char **) realloc(ret, sizeof(char*) * maxCount);
        }
    }

error:
    for (i = 0; i < *count; i++) {
        free(ret[i]);
    }
    free(ret);
    ret = NULL;
    *count = 0;
done:
//    ret = (char **) realloc(ret, sizeof(char**) * (*count));
    sqlite3_reset(ch->uniqFilesStatement);

    return ret;
}
int cacheGet(
    CacheHandle_t* ch,
    const char* key,
    void** value,
    int* size,
    int* decompressedSize)
{
    int ret = -1;
    int rc = sqlite3_bind_text(
        ch->getStatement,
        1,
        key,
        -1,
        0);
    if (rc != SQLITE_OK) {
        setLastError(
            ch,
            "cacheGet.bind_text_1",
            rc);
        goto exit;
    }
    rc = sqlite3_step(ch->getStatement);
    if (rc != SQLITE_ROW) {
        if (rc == SQLITE_DONE) {
            *value = NULL;
            *size = -1;
            ret = 0;
            goto exit;
        }
        setLastError(
            ch,
            "cacheGet.step",
            rc);
        goto exit;
    }

    *value = NULL;
    const void* valueBlob = sqlite3_column_blob(ch->getStatement, 0);
    *size = sqlite3_column_bytes(ch->getStatement, 0);
    *decompressedSize = sqlite3_column_int(ch->getStatement, 1);

    if (valueBlob != NULL && *size != 0) {
        *value = (void *) malloc(*size);
        memcpy(*value, valueBlob, *size);
//        for (int i = 0; i < 100; i++) {
//            fprintf(stderr, " %x", ((char*)*value)[i]);
//        }
//        fprintf(stderr, "\n");
    }

    ret = 0;

exit:
    sqlite3_reset(ch->getStatement);

    return ret;
}

int cacheGetOne(
    CacheHandle_t* ch,
    void** value,
    char** key,
    int* size,
    int* decompressedSize,
    int* accessFreq)
{
    int ret = -1;
    int rc = sqlite3_step(ch->getOneStatement);
    if (rc != SQLITE_ROW) {
        if (rc == SQLITE_DONE) {
            *value = NULL;
            *size = -1;
            ret = 0;
            goto exit;
        }
        setLastError(
            ch,
            "cacheGetOne.step",
            rc);
        goto exit;
    }

    *value = NULL;
    const void* valueBlob = sqlite3_column_blob(ch->getOneStatement, 0);
    *size = sqlite3_column_bytes(ch->getOneStatement, 0);
    *decompressedSize = sqlite3_column_int(ch->getOneStatement, 1);
    *accessFreq = sqlite3_column_int(ch->getOneStatement, 2);
    *key = strdup((const char*) sqlite3_column_text(ch->getOneStatement, 3));

    if (valueBlob != NULL && *size != 0) {
        *value = (void *) malloc(*size);
        memcpy(*value, valueBlob, *size);
//        for (int i = 0; i < 100; i++) {
//            fprintf(stderr, " %x", ((char*)*value)[i]);
//        }
//        fprintf(stderr, "\n");
    }

    ret = 0;

exit:
    sqlite3_reset(ch->getOneStatement);

    return ret;
}

int dropLeast(CacheHandle_t* ch, long toSize) {
    long size = cacheSize(ch);
    if (size == -1) {
        return -1;
    }
    while (size > toSize) {
        if (deleteTop(ch, 100)) {
            return -1;
        }
        size = cacheSize(ch);
        if (size == -1) {
            return -1;
        }
    }
    return 0;
}

CacheHandle_t* initCache(const char* path, jboolean shared) {
    CacheHandle_t *handle = malloc(sizeof(CacheHandle_t));
    handle->lastErrorBufferSize = 1024;
    handle->lastError = (char *) malloc (handle->lastErrorBufferSize);
    handle->lastError[0] = 0;
    if (openDatabase(handle, path, shared)) {
        handle->db = NULL;
        goto exit;
    }
    if (createTables(handle)) {
        handle->db = NULL;
        goto exit;
    }
    if (prepareStatements(handle)) {
        handle->db = NULL;
        goto exit;
    }
exit:
    return handle;
}

void closeHandle(CacheHandle_t* handle) {
    sqlite3_close(handle->db);
    free(handle->lastError);
    free(handle);
}

void freeBlob(void* blob) {
    free(blob);
}

unsigned int crc32_for_byte(unsigned int r) {
    for(int j = 0; j < 8; ++j) {
        r = (r & 1? 0: (unsigned int)0xEDB88320L) ^ r >> 1;
    }
    return r ^ (unsigned int)0xFF000000L;
}

void crc32(const void *data, size_t n_bytes, unsigned int* crc) {
    static unsigned int table[0x100];
    if(!*table) {
        for(size_t i = 0; i < 0x100; ++i) {
            table[i] = crc32_for_byte(i);
        }
    }
    for(size_t i = 0; i < n_bytes; ++i) {
        *crc = table[(unsigned char)*crc ^ ((unsigned char*)data)[i]] ^ *crc >> 8;
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_init
(JNIEnv* env,
jclass class,
jstring dbPath,
jboolean shared)
{
    (void) class;
    const char *nativeString = (*env)->GetStringUTFChars(env, dbPath, 0);
    CacheHandle_t *handle = initCache(nativeString, shared);
    (*env)->ReleaseStringUTFChars(env, dbPath, nativeString);
    if (handle->db == NULL) {
        jclass exClass;
        char *className = "java/io/IOException";
        exClass = (*env)->FindClass(env, className);
        if (exClass == NULL) {
            return 0;
        }
        return (*env)->ThrowNew(env, exClass, handle->lastError);
    } else {
        return (jlong) handle;
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_commonInit
(JNIEnv* env,
jclass class)
{
    (void) class;
    CacheHandle_t *handle = malloc(sizeof(CacheHandle_t));
    handle->lastErrorBufferSize = 1024;
    handle->lastError = (char *) malloc (handle->lastErrorBufferSize);
    handle->lastError[0] = 0;
    int ret = commonInit(handle);
    if (ret) {
        jclass exClass;
        char *className = "java/io/IOException";
        exClass = (*env)->FindClass(env, className);
        (*env)->ThrowNew(env, exClass, handle->lastError);
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_put
(JNIEnv* env,
jclass class,
jlong handle,
jstring key,
jlong value,
jint len,
jint decompressedSize,
jint accessFreq)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char *keyChars = (*env)->GetStringUTFChars(env, key, 0);
    int ret =
        cachePut(
            ch,
            keyChars,
            (const void*) value,
            len,
            decompressedSize,
            accessFreq);
    (*env)->ReleaseStringUTFChars(env, key, keyChars);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_update
(JNIEnv* env,
jclass class,
jlong handle,
jstring key)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char *keyChars = (*env)->GetStringUTFChars(env, key, 0);
    int ret = cacheUpdate(ch, keyChars);
    (*env)->ReleaseStringUTFChars(env, key, keyChars);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_remove
(JNIEnv* env,
jclass class,
jlong handle,
jstring key)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char *keyChars = (*env)->GetStringUTFChars(env, key, 0);
    int ret = cacheRemove(ch, keyChars);
    (*env)->ReleaseStringUTFChars(env, key, keyChars);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_removePrefix
(JNIEnv* env,
jclass class,
jlong handle,
jstring prefix)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char *prefixChars = (*env)->GetStringUTFChars(env, prefix, 0);
    int ret = cacheRemovePrefix(ch, prefixChars);
    (*env)->ReleaseStringUTFChars(env, prefix, prefixChars);
    return ret;
}

__attribute__((__visibility__("default")))
JNIEXPORT jobject JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_get
(JNIEnv* env,
jclass class,
jlong handle,
jstring key)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    const char* keyChars = (*env)->GetStringUTFChars(env, key, 0);
    void* value = 0;
    int size = -1;
    int decompressedSize = 0;
    int ret = cacheGet(ch, keyChars, &value, &size, &decompressedSize);
    (*env)->ReleaseStringUTFChars(env, key, keyChars);
    if (ret) {
        return NULL;
    }

    jobject cls;
    jmethodID constructor;
    jobject object;

    cls =
        (*env)->FindClass(env, "ru/yandex/cache/sqlite/SqliteCacheEntry");
    if (cls == NULL) {
        fprintf(stderr, "class not found\n");
    }
    constructor = (*env)->GetMethodID(env, cls, "<init>", "(IIJ)V");
    if (constructor == NULL) {
        fprintf(stderr, "constructor not found\n");
    }
    object = (*env)->NewObject(
        env,
        cls,
        constructor,
        size,
        decompressedSize,
        (jlong) value);

    return object;
}

__attribute__((__visibility__("default")))
JNIEXPORT jobject JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_getOne
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    char* key = 0;
    void* value = 0;
    int size = -1;
    int decompressedSize = 0;
    int accessFreq = 0;
    int ret = cacheGetOne(
        ch,
        &value,
        &key,
        &size,
        &decompressedSize,
        &accessFreq);
    if (ret) {
        return NULL;
    }

    jobject cls;
    jmethodID constructor;
    jobject object;

    cls =
        (*env)->FindClass(env, "ru/yandex/cache/sqlite/SqliteCopyEntry");
    if (cls == NULL) {
        fprintf(stderr, "class not found\n");
    }
    constructor = (*env)->GetMethodID(
        env,
        cls,
        "<init>",
        "(Ljava/lang/String;IIIJ)V");

    if (constructor == NULL) {
        fprintf(stderr, "constructor not found\n");
    }
    object = (*env)->NewObject(
        env,
        cls,
        constructor,
        (*env)->NewStringUTF(env, key),
        size,
        decompressedSize,
        accessFreq,
        (jlong) value);
    free(key);

    return object;
}

__attribute__((__visibility__("default")))
JNIEXPORT jstring JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_lastError
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return (*env)->NewStringUTF(env, ch->lastError);
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_cacheSize
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    (void) env;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return cacheSize(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL JavaCritical_ru_yandex_cache_sqlite_SqliteCache_cacheSize
(jlong handle)
{
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return cacheSize(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_deleteTop
(JNIEnv* env,
jclass class,
jlong handle,
jint count)
{
    (void) class;
    (void) env;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return deleteTop(ch, count);
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL JavaCritical_ru_yandex_cache_sqlite_SqliteCache_deleteTop
(jlong handle,
jint count)
{
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return deleteTop(ch, count);
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_closeHandle
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    (void) env;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return closeHandle(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL JavaCritical_ru_yandex_cache_sqlite_SqliteCache_closeHandle
(jlong handle)
{
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    return closeHandle(ch);
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_freeBlob
(JNIEnv* env,
jclass class,
jlong blob)
{
    (void) class;
    (void) env;
    freeBlob((void *) blob);
}

__attribute__((__visibility__("default")))
JNIEXPORT void JNICALL JavaCritical_ru_yandex_cache_sqlite_SqliteCache_freeBlob
(jlong blob)
{
    freeBlob((void *) blob);
}


__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_copyBlob
(JNIEnv* env,
jclass class,
jlong blob,
jint len)
{
    (void) class;
    (void) env;
    if (blob != 0 && len > 0) {
        void* newBlob = (void*) malloc(len);
        memcpy(newBlob, (void*) blob, len);
        return (jlong) newBlob;
    } else {
        return 0;
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jlong JNICALL JavaCritical_ru_yandex_cache_sqlite_SqliteCache_copyBlob
(jlong blob, jint len)
{
    if (blob != 0 && len > 0) {
        void* newBlob = (void*) malloc(len);
        memcpy(newBlob, (void*) blob, len);
        return (jlong) newBlob;
    } else {
        return 0;
    }
}

__attribute__((__visibility__("default")))
JNIEXPORT jint JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_crc32
(JNIEnv* env,
jclass class,
jlong blob,
jint len)
{
    (void) class;
    (void) env;
    unsigned int crc = 0;
    crc32((void*) blob, len, &crc);
    return (jint) crc;
}

__attribute__((__visibility__("default")))
JNIEXPORT jobjectArray JNICALL Java_ru_yandex_cache_sqlite_SqliteCache_uniqFiles
(JNIEnv* env,
jclass class,
jlong handle)
{
    (void) class;
    CacheHandle_t* ch = (CacheHandle_t*) handle;
    int fileCount = 0;
    char **files = cacheUniqFiles(ch, &fileCount);
//    char **files = NULL;
    jobjectArray ret = NULL;
    if (files == NULL) {
        return ret;
    }
    ret= (jobjectArray) (*env)->NewObjectArray(
        env,
        fileCount,
        (*env)->FindClass(env, "java/lang/String"),
        (*env)->NewStringUTF(env, ""));
    int i;
    for (i = 0; i < fileCount; i++) {
        (*env)->SetObjectArrayElement(
            env,
            ret,
            i,
            (*env)->NewStringUTF(env, files[i]));
        free(files[i]);
    }
    free(files);
    return ret;
}

/*
int main(int argc, char** argv) {
    sqlite3* db;
    char* zErrMsg = 0;

    if (db == NULL) {
        db = openDatabase(":memory:");
    }

    if (db == NULL) {
        return 1;
    }

    if (createTables(db)) {
        goto ERROR;
    }

    if (putStatement == NULL) {
        if (prepareStatements(db)) {
            goto ERROR;
        }
    }

    int i;
    char *buf = malloc(8192);
    for (i = 0; i < 8192; i++) {
        buf[i] = i & 0xFF;
    }
    char *key = malloc(1024);
    for (i = 0; i < 1000000; i++) {
        long size = cacheSize(db);
        printf("\rsize: %010ld", size);
        snprintf(key, 1023, "/qwe/sdf/zcqwe/%d:%x", i, i);
        cachePut(db, key, buf, 8192 - (i % 1024));
        if (size == -1) {
            goto ERROR;
        }
        if (size > 1 * 1024 * 1024 * 1024) {
            int rc = dropLeast(
                db,
                1 * 1024 * 1024 * 1024
                    - 128 * 1024 * 1024);
            if (rc == -1) {
                goto ERROR;
            }
        }
    }

    int t;
    for (t = 0; t < 100; t++) {
        printf("\nTest #%d\n", t);
        for (i = 0; i < 100000; i += 4) {
            printf("\rsize: %010ld", cacheSize(db));
            snprintf(key, 1023, "/qwe/sdf/zcqwe/%d:%x", i, i);
//            printf("Removing %s\n", key);
            cacheRemove(db, key);
        }

        for (i = 0; i < 100000; i += 4) {
            printf("\rsize: %010ld", cacheSize(db));
            snprintf(key, 1023, "/qwe/sdf/zcqwe/%d:%x", i, i);
            cachePut(db, key, buf, 8192 - (i % 1024));
        }
    }

    sqlite3_close(db);
    return 0;

ERROR:
    sqlite3_close(db);
    return 1;
}
*/