package ru.yandex.util.storage;

import java.io.IOException;
import java.io.InputStream;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;

import ru.yandex.http.util.PayloadTooLargeException;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.io.LimitedIOException;
import ru.yandex.io.LimitedOutputStream;

public enum RawDataExtractor implements DataExtractor {
    INSTANCE;

    @Override
    public StorageData extractData(
        final InputStream in,
        final long contentLength,
        final ImmutableDataExtractorConfig config)
        throws HttpException, IOException
    {
        DecodableByteArrayOutputStream out;
        if (contentLength == -1L) {
            out = new DecodableByteArrayOutputStream();
            LimitedOutputStream limiter =
                new LimitedOutputStream(out, config.maxInputLength());
            try {
                IOStreamUtils.copy(in, limiter);
            } catch (LimitedIOException e) {
                if (!config.truncateLongInput()) {
                    throw new PayloadTooLargeException();
                }
            }
        } else if (contentLength > config.maxInputLength()) {
            if (config.truncateLongInput()) {
                out = new DecodableByteArrayOutputStream(
                    config.maxInputLength());
                LimitedOutputStream limiter =
                    new LimitedOutputStream(out, config.maxInputLength());
                try {
                    IOStreamUtils.copy(in, limiter);
                } catch (LimitedIOException e) {
                    // It's ok
                }
            } else {
                throw new PayloadTooLargeException(
                    "Input length is " + contentLength
                    + " while the max length is " + config.maxInputLength());
            }
        } else {
            out = new DecodableByteArrayOutputStream((int) contentLength);
            IOStreamUtils.copy(in, out);
        }
        return new StorageData(null, out);
    }

    @Override
    public StorageData extractData(
        final HttpResponse response,
        final ImmutableDataExtractorConfig config)
        throws HttpException, IOException
    {
        HttpEntity entity = response.getEntity();
        long contentLength = entity.getContentLength();
        DecodableByteArrayOutputStream out;
        if (contentLength == -1L) {
            out = new DecodableByteArrayOutputStream();
            LimitedOutputStream limiter =
                new LimitedOutputStream(out, config.maxInputLength());
            try {
                entity.writeTo(limiter);
            } catch (LimitedIOException e) {
                if (!config.truncateLongInput()) {
                    throw new PayloadTooLargeException();
                }
            }
        } else if (contentLength > config.maxInputLength()) {
            if (config.truncateLongInput()) {
                out = new DecodableByteArrayOutputStream(
                    config.maxInputLength());
                LimitedOutputStream limiter =
                    new LimitedOutputStream(out, config.maxInputLength());
                try {
                    entity.writeTo(limiter);
                } catch (LimitedIOException e) {
                    // It's ok
                }
            } else {
                throw new PayloadTooLargeException(
                    "Response length is " + contentLength
                    + " while the max input length is "
                    + config.maxInputLength());
            }
        } else {
            out = new DecodableByteArrayOutputStream((int) contentLength);
            entity.writeTo(out);
        }
        return new StorageData(null, out);
    }
}

