package ru.yandex.util.storage;

import java.io.IOException;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.RequestToString;
import ru.yandex.http.util.SynchronizedHttpContext;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.client.Timings;
import ru.yandex.io.GenericCloseableAdapter;
import ru.yandex.util.string.StringUtils;

public class StorageClient implements GenericAutoCloseable<IOException> {
    private final CloseableHttpClient client;
    private final String uriSuffix;

    public StorageClient(
        final CloseableHttpClient client,
        final String uriSuffix)
    {
        this.client = client;
        this.uriSuffix = uriSuffix;
    }

    @Override
    public void close() throws IOException {
        client.close();
    }

    // CSOFF: ParameterNumber
    public CloseableHttpResponse sendStorageRequest(
        final HttpHost storageHost,
        final String query,
        final Logger logger,
        final Header... headers)
        throws HttpException, IOException
    {
        return sendStorageRequest(
            storageHost,
            query,
            logger,
            new HttpClientContext(new SynchronizedHttpContext()),
            headers);
    }

    public CloseableHttpResponse sendStorageRequest(
        final HttpHost storageHost,
        final String query,
        final Logger logger,
        final HttpClientContext context,
        final Header... headers)
        throws HttpException, IOException
    {
        String uri;
        if (uriSuffix == null) {
            uri = query;
        } else if (query.indexOf('?') == -1) {
            uri = StringUtils.concat(query, '?', uriSuffix);
        } else {
            uri = StringUtils.concat(query, '&', uriSuffix);
        }
        HttpRequest request = new BasicHttpRequest(HttpGet.METHOD_NAME, uri);
        request.setHeaders(headers);
        Timings timings = new Timings();
        context.setAttribute(Timings.TIMINGS, timings);
        try (GenericAutoCloseableHolder<
                IOException,
                GenericCloseableAdapter<CloseableHttpResponse>> holder =
                    new GenericAutoCloseableHolder<>(
                        new GenericCloseableAdapter<>(
                            client.execute(
                                storageHost,
                                request,
                                context))))
        {
            CloseableHttpResponse response = holder.get().get();
            StringBuilder sb = new StringBuilder("Storage responded for ");
            sb.append(storageHost.getSchemeName());
            sb.append("://");
            sb.append(storageHost.getHostName());
            sb.append(':');
            sb.append(storageHost.getPort());
            sb.append(query);
            sb.append(" with ");
            sb.append(response.getStatusLine());
            sb.append(' ');
            RequestToString.traverseHeaders(
                new RequestToString.HeadersPrinter(
                    sb,
                    YandexHeaders.DEFAULT_HIDDEN_HEADERS),
                response.getAllHeaders());
            sb.append(' ');
            sb.append(response.getEntity());
            sb.append(" in ");
            timings.toStringBuilder(sb);
            logger.info(new String(sb));
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_OK) {
                HttpEntity entity = response.getEntity();
                if (entity == null || entity.getContent() == null) {
                    throw new BadGatewayException("Empty response");
                } else {
                    return holder.release().get();
                }
            } else {
                HttpException e = new BadResponseException(request, response);
                if (status == HttpStatus.SC_FORBIDDEN) {
                    // Do not return status code 403 to client as it could
                    // prevent other hosts checking
                    e = new BadGatewayException(e);
                }
                throw e;
            }
        }
    }
    // CSON: ParameterNumber
}

