package ru.yandex.tikaite.config;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.mail.received.ImmutableReceivedChainParserConfig;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.sanitizer2.config.SanitizingConfig;

public class ImmutableTextExtractorConfig implements TextExtractorConfig {
    private final long tmpFileLimit;
    private final long memoryLimit;
    private final Set<Integer> noXurlsTypes;
    private final int diskLengthLimit;
    private final int mailLengthLimit;
    private final int textLengthLimit;
    private final int headersLengthLimit;
    private final int headerLengthLimit;
    private final int maxHidDepth;
    private final int maxParts;
    private final int maxPartLength;
    private final ImmutableSanitizingConfig sanitizingConfig;
    private final int maxSanitizingLength;
    private final ImmutableFieldsFilterConfig fieldsFilterConfig;
    private final ImmutableReceivedChainParserConfig receivedChainParserConfig;

    public ImmutableTextExtractorConfig(final TextExtractorConfig config)
        throws ConfigException
    {
        tmpFileLimit = config.tmpFileLimit();
        memoryLimit = config.memoryLimit();
        noXurlsTypes =
            Collections.unmodifiableSet(
                new HashSet<>(config.noXurlsTypes()));
        diskLengthLimit = config.diskLengthLimit();
        mailLengthLimit = config.mailLengthLimit();
        textLengthLimit = config.textLengthLimit();
        headersLengthLimit = config.headersLengthLimit();
        headerLengthLimit = config.headerLengthLimit();
        maxHidDepth = config.maxHidDepth();
        maxParts = config.maxParts();
        maxPartLength = config.maxPartLength();
        SanitizingConfig sanitizingConfig = config.sanitizingConfig();
        if (sanitizingConfig == null) {
            this.sanitizingConfig = null;
        } else {
            this.sanitizingConfig =
                new ImmutableSanitizingConfig(sanitizingConfig);
        }
        maxSanitizingLength = config.maxSanitizingLength();

        FieldsFilterConfig fieldsFilterConfig = config.fieldsFilterConfig();
        if (fieldsFilterConfig == null) {
            this.fieldsFilterConfig = null;
        } else {
            this.fieldsFilterConfig =
                new ImmutableFieldsFilterConfig(fieldsFilterConfig);
        }

        receivedChainParserConfig = RECEIVED_CHAIN_PARSER.validate(
            config.receivedChainParserConfig());

        if (tmpFileLimit < 0L) {
            throw new ConfigException(
                "Temporary file size limit must be non-negative");
        }
        if (memoryLimit < 0L) {
            throw new ConfigException(
                "Memory usage limit must be non-negative");
        }
        if (maxHidDepth <= 0) {
            throw new ConfigException("Max hid depth must be positive");
        }
    }

    @Override
    public long tmpFileLimit() {
        return tmpFileLimit;
    }

    @Override
    public long memoryLimit() {
        return memoryLimit;
    }

    @Override
    public Set<Integer> noXurlsTypes() {
        return noXurlsTypes;
    }

    @Override
    public int diskLengthLimit() {
        return diskLengthLimit;
    }

    @Override
    public int mailLengthLimit() {
        return mailLengthLimit;
    }

    @Override
    public int textLengthLimit() {
        return textLengthLimit;
    }

    @Override
    public int headersLengthLimit() {
        return headersLengthLimit;
    }

    @Override
    public int headerLengthLimit() {
        return headerLengthLimit;
    }

    @Override
    public int maxHidDepth() {
        return maxHidDepth;
    }

    @Override
    public int maxParts() {
        return maxParts;
    }

    @Override
    public int maxPartLength() {
        return maxPartLength;
    }

    @Override
    public ImmutableSanitizingConfig sanitizingConfig() {
        return sanitizingConfig;
    }

    @Override
    public int maxSanitizingLength() {
        return maxSanitizingLength;
    }

    @Override
    public ImmutableFieldsFilterConfig fieldsFilterConfig() {
        return fieldsFilterConfig;
    }

    @Override
    public ImmutableReceivedChainParserConfig receivedChainParserConfig() {
        return receivedChainParserConfig;
    }
}

