package ru.yandex.tikaite.config;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.mail.received.ReceivedChainParserConfig;
import ru.yandex.parser.mail.received.ReceivedChainParserConfigBuilder;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.LongMemorySizeParser;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.sanitizer2.config.SanitizingConfig;
import ru.yandex.sanitizer2.config.SanitizingConfigBuilder;

public class TextExtractorConfigBuilder implements TextExtractorConfig {
    private long tmpFileLimit;
    private long memoryLimit;
    private Set<Integer> noXurlsTypes;
    private int diskLengthLimit;
    private int mailLengthLimit;
    private int textLengthLimit;
    private int headersLengthLimit;
    private int headerLengthLimit;
    private int maxHidDepth;
    private int maxParts;
    private int maxPartLength;
    private SanitizingConfigBuilder sanitizingConfig;
    private int maxSanitizingLength;
    private FieldsFilterConfigBuilder fieldsFilterConfig;
    private ReceivedChainParserConfigBuilder receivedChainParserConfig;

    public TextExtractorConfigBuilder() {
        this(TextExtractorConfigDefaults.INSTANCE);
    }

    public TextExtractorConfigBuilder(final TextExtractorConfig config) {
        tmpFileLimit(config.tmpFileLimit());
        memoryLimit(config.memoryLimit());
        noXurlsTypes(config.noXurlsTypes());
        diskLengthLimit(config.diskLengthLimit());
        mailLengthLimit(config.mailLengthLimit());
        textLengthLimit(config.textLengthLimit());
        headersLengthLimit(config.headersLengthLimit());
        headerLengthLimit(config.headerLengthLimit());
        maxHidDepth(config.maxHidDepth());
        maxParts(config.maxParts());
        maxPartLength(config.maxPartLength());
        sanitizingConfig(config.sanitizingConfig());
        maxSanitizingLength(config.maxSanitizingLength());
        fieldsFilterConfig(config.fieldsFilterConfig());
        receivedChainParserConfig(config.receivedChainParserConfig());
    }

    public TextExtractorConfigBuilder(final IniConfig config)
        throws ConfigException
    {
        this(config, TextExtractorConfigDefaults.INSTANCE);
    }

    public TextExtractorConfigBuilder(
        final IniConfig config,
        final TextExtractorConfig defaults)
        throws ConfigException
    {
        tmpFileLimit = config.get(
            "tmp-file-limit",
            defaults.tmpFileLimit(),
            LongMemorySizeParser.INSTANCE);
        memoryLimit = config.get(
            "memory-limit",
            defaults.memoryLimit(),
            LongMemorySizeParser.INSTANCE);
        noXurlsTypes = config.get(
            "no-xurls-types",
            defaults.noXurlsTypes(),
            new CollectionParser<>(
                PositiveIntegerValidator.INSTANCE.compose(String::trim),
                HashSet::new));
        diskLengthLimit =
            config.getInt("disk-length-limit", defaults.diskLengthLimit());
        mailLengthLimit =
            config.getInt("mail-length-limit", defaults.mailLengthLimit());
        textLengthLimit =
            config.getInt("text-length-limit", defaults.textLengthLimit());
        headersLengthLimit = config.getInt(
            "headers-length-limit",
            defaults.headersLengthLimit());
        headerLengthLimit =
            config.getInt("header-length-limit", defaults.headerLengthLimit());
        maxHidDepth = config.get(
            "max-hid-depth",
            defaults.maxHidDepth(),
            PositiveIntegerValidator.INSTANCE);
        maxParts = config.get(
            "max-parts",
            defaults.maxParts(),
            PositiveIntegerValidator.INSTANCE);
        maxPartLength =
            config.getInt("max-part-length", defaults.maxPartLength());

        File sanitizingConfig = config.getInputFile("sanitizing-config", null);
        if (sanitizingConfig == null) {
            this.sanitizingConfig = null;
        } else {
            try {
                this.sanitizingConfig = new SanitizingConfigBuilder(
                    new IniConfig(sanitizingConfig));
            } catch (IOException e) {
                throw new ConfigException(
                    "Failed to load sanitizing config from file "
                    + sanitizingConfig,
                    e);
            }
        }

        maxSanitizingLength = config.getInt(
            "max-sanitizing-length",
            defaults.maxSanitizingLength());

        IniConfig fieldsFilterSection = config.sectionOrNull("fields-filter");
        FieldsFilterConfig fieldsFilterConfigDefaults =
            defaults.fieldsFilterConfig();
        if (fieldsFilterSection == null) {
            fieldsFilterConfig(fieldsFilterConfigDefaults);
        } else if (fieldsFilterConfigDefaults == null) {
            fieldsFilterConfig =
                new FieldsFilterConfigBuilder(fieldsFilterSection);
        } else {
            fieldsFilterConfig = new FieldsFilterConfigBuilder(
                fieldsFilterSection,
                fieldsFilterConfigDefaults);
        }

        receivedChainParserConfig = RECEIVED_CHAIN_PARSER.extract(
            config,
            defaults.receivedChainParserConfig());
    }

    @Override
    public long tmpFileLimit() {
        return tmpFileLimit;
    }

    public TextExtractorConfigBuilder tmpFileLimit(final long tmpFileLimit) {
        this.tmpFileLimit = tmpFileLimit;
        return this;
    }

    @Override
    public long memoryLimit() {
        return memoryLimit;
    }

    public TextExtractorConfigBuilder memoryLimit(final long memoryLimit) {
        this.memoryLimit = memoryLimit;
        return this;
    }

    @Override
    public Set<Integer> noXurlsTypes() {
        return noXurlsTypes;
    }

    public TextExtractorConfigBuilder noXurlsTypes(
        final Set<Integer> noXurlsTypes)
    {
        this.noXurlsTypes = noXurlsTypes;
        return this;
    }

    @Override
    public int diskLengthLimit() {
        return diskLengthLimit;
    }

    public TextExtractorConfigBuilder diskLengthLimit(
        final int diskLengthLimit)
    {
        this.diskLengthLimit = diskLengthLimit;
        return this;
    }

    @Override
    public int mailLengthLimit() {
        return mailLengthLimit;
    }

    public TextExtractorConfigBuilder mailLengthLimit(
        final int mailLengthLimit)
    {
        this.mailLengthLimit = mailLengthLimit;
        return this;
    }

    @Override
    public int textLengthLimit() {
        return textLengthLimit;
    }

    public TextExtractorConfigBuilder textLengthLimit(
        final int textLengthLimit)
    {
        this.textLengthLimit = textLengthLimit;
        return this;
    }

    @Override
    public int headersLengthLimit() {
        return headersLengthLimit;
    }

    public TextExtractorConfigBuilder headersLengthLimit(
        final int headersLengthLimit)
    {
        this.headersLengthLimit = headersLengthLimit;
        return this;
    }

    @Override
    public int headerLengthLimit() {
        return headerLengthLimit;
    }

    public TextExtractorConfigBuilder headerLengthLimit(
        final int headerLengthLimit)
    {
        this.headerLengthLimit = headerLengthLimit;
        return this;
    }

    @Override
    public int maxHidDepth() {
        return maxHidDepth;
    }

    public TextExtractorConfigBuilder maxHidDepth(final int maxHidDepth) {
        this.maxHidDepth = maxHidDepth;
        return this;
    }

    @Override
    public int maxParts() {
        return maxParts;
    }

    public TextExtractorConfigBuilder maxParts(final int maxParts) {
        this.maxParts = maxParts;
        return this;
    }

    @Override
    public int maxPartLength() {
        return maxPartLength;
    }

    public TextExtractorConfigBuilder maxPartLength(final int maxPartLength) {
        this.maxPartLength = maxPartLength;
        return this;
    }

    @Override
    public SanitizingConfigBuilder sanitizingConfig() {
        return sanitizingConfig;
    }

    public TextExtractorConfigBuilder sanitizingConfig(
        final SanitizingConfig sanitizingConfig)
    {
        if (sanitizingConfig == null) {
            this.sanitizingConfig = null;
        } else {
            this.sanitizingConfig =
                new SanitizingConfigBuilder(sanitizingConfig);
        }
        return this;
    }

    @Override
    public int maxSanitizingLength() {
        return maxSanitizingLength;
    }

    public TextExtractorConfigBuilder maxSanitizingLength(
        final int maxSanitizingLength)
    {
        this.maxSanitizingLength = maxSanitizingLength;
        return this;
    }

    @Override
    public FieldsFilterConfigBuilder fieldsFilterConfig() {
        return fieldsFilterConfig;
    }

    public TextExtractorConfigBuilder fieldsFilterConfig(
        final FieldsFilterConfig fieldsFilterConfig)
    {
        if (fieldsFilterConfig == null) {
            this.fieldsFilterConfig = null;
        } else {
            this.fieldsFilterConfig =
                new FieldsFilterConfigBuilder(fieldsFilterConfig);
        }
        return this;
    }

    @Override
    public ReceivedChainParserConfigBuilder receivedChainParserConfig() {
        return receivedChainParserConfig;
    }

    public TextExtractorConfigBuilder receivedChainParserConfig(
        final ReceivedChainParserConfig receivedChainParserConfig)
    {
        this.receivedChainParserConfig =
            new ReceivedChainParserConfigBuilder(receivedChainParserConfig);
        return this;
    }

    public ImmutableTextExtractorConfig build() throws ConfigException {
        return new ImmutableTextExtractorConfig(this);
    }
}

