package ru.yandex.tikaite.detect;

import java.io.IOException;
import java.io.InputStream;

import org.apache.tika.detect.Detector;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.mime.MediaType;

public enum TarDetector implements Detector {
    INSTANCE;

    private static final int PREFIX_LENGTH = 148;
    private static final int CHKSUM_LENGTH = 8;
    private static final int SUFFIX_LENGTH = 356;
    private static final int OCTAL_SHIFT = 3;
    private static final MediaType TAR = MediaType.application("x-tar");

    // CSOFF: ReturnCount
    @Override
    public MediaType detect(final InputStream in, final Metadata metadata)
        throws IOException
    {
        in.mark(PREFIX_LENGTH + CHKSUM_LENGTH + SUFFIX_LENGTH);
        try {
            int sum = 0;
            for (int i = 0; i < PREFIX_LENGTH; ++i) {
                int read = in.read();
                if (read == -1) {
                    return MediaType.OCTET_STREAM;
                }
                sum += read;
            }
            if (sum == 0) {
                return MediaType.OCTET_STREAM;
            }
            sum += ' ' * CHKSUM_LENGTH;
            boolean begin = true;
            boolean end = false;
            int chksum = -1;
            for (int i = 0; i < CHKSUM_LENGTH; ++i) {
                int read = in.read();
                if (read == -1) {
                    return MediaType.OCTET_STREAM;
                }
                if (end) {
                    if (read != ' ' && read != 0) {
                        return MediaType.OCTET_STREAM;
                    }
                } else {
                    if (begin) {
                        if (read != ' ') {
                            begin = false;
                            chksum = 0;
                        }
                    }
                    if (!begin) {
                        if (read >= '0' && read < '8') {
                            chksum <<= OCTAL_SHIFT;
                            chksum |= read - '0';
                        } else {
                            end = true;
                        }
                    }
                }
            }
            if (chksum == -1) {
                return MediaType.OCTET_STREAM;
            }
            for (int i = 0; i < SUFFIX_LENGTH; ++i) {
                int read = in.read();
                if (read == -1) {
                    return MediaType.OCTET_STREAM;
                }
                sum += read;
            }
            if (sum == chksum) {
                return TAR;
            } else {
                return MediaType.OCTET_STREAM;
            }
        } finally {
            in.reset();
        }
    }
    // CSON: ReturnCount
}

