package ru.yandex.tikaite.mimeparser;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;

import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.parser.MimeStreamParser;

import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.document.mail.SafeMimeTokenStream;
import ru.yandex.tikaite.util.TextExtractOptions;

public class EnclosedParser implements Closeable {
    private static final byte[] SKIP_MARK =
        "\n</message>\n".getBytes(StandardCharsets.UTF_8);

    private final MailMetaInfo meta;
    private final InputStream in;
    private final TextExtractOptions options;
    private final SafeMimeTokenStream mimeTokenStream;
    private final MimeStreamParser mimeStreamParser;

    public EnclosedParser(
        final MailMetaInfo meta,
        final InputStream in,
        final TextExtractOptions options)
        throws IOException, MimeException
    {
        this.meta = meta;
        this.in = new BufferedInputStream(in);
        this.in.mark(1);
        int first = this.in.read();
        if (first == -1) {
            throw new EOFException("Empty stream");
        }
        this.in.reset();
        if (first == '<') {
            IOStreamUtils.skipTo(this.in, SKIP_MARK);
        }
        mimeTokenStream = SafeMimeTokenStream.parseMeta(this.in, meta);
        this.options =
            new TextExtractOptions(options)
                .mimeTokenStream(mimeTokenStream);
        mimeStreamParser = new MimeStreamParser(mimeTokenStream);
    }

    public void parse(final JsonWriterBase writer)
        throws IOException, MimeException
    {
        ManagedHandler handler = new RootMessageHandler(
            meta,
            options,
            new MailDocumentCollectorFactory(
                new JsonDocumentCollectorFactory(
                    writer,
                    options.fieldsFilter(),
                    options.fieldsFilterOptions())));
        HandlerManager manager = new HandlerManager();
        manager.push(handler);
        mimeStreamParser.setContentHandler(manager);
        mimeStreamParser.parse(null);
    }

    @Override
    public void close() throws IOException {
        in.close();
    }
}

