package ru.yandex.tikaite.mimeparser;

import java.io.IOException;
import java.io.InputStream;

import com.twmacinta.util.MD5InputStream;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.BodyDescriptor;

import ru.yandex.io.CountingInputStream;
import ru.yandex.io.LimitedInputStream;
import ru.yandex.io.TaggedInputStream;
import ru.yandex.mail.mime.BodyDecoder;
import ru.yandex.search.document.mail.HidCounter;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.tikaite.util.TextExtractOptions;
import ru.yandex.util.string.HexStrings;

public abstract class MessageHandler extends EmptyMessageHandler {
    protected final TextExtractOptions options;

    protected MessageHandler(
        final MailMetaInfo meta,
        final TextExtractOptions options)
    {
        super(meta);
        this.options = options;
    }

    protected abstract DocumentCollector collector() throws IOException;

    @Override
    public void startMessage() throws MimeException {
        try {
            manager.push(new NestedMessageHandler(collector(), meta, options));
        } catch (IOException e) {
            throw new MimeException("Failed to create collector", e);
        }
    }

    @Override
    public void endMessage() throws MimeException {
        manager.pop();
    }

    @Override
    public void startMultipart(final BodyDescriptor bd) throws MimeException {
        try {
            manager.push(new MultipartHandler(collector(), meta, options));
        } catch (IOException e) {
            throw new MimeException("Failed to create collector", e);
        }
    }

    protected void extractText(
        final String hid,
        final BodyDescriptor bd,
        final InputStream is)
        throws IOException
    {
        try (TaggedInputStream tagged = new TaggedInputStream(is);
            MailDocument document =
                collector().createDocument(meta, hid, options))
        {
            TaggedInputStream decoded = new TaggedInputStream(
                BodyDecoder.INSTANCE.apply(tagged, bd.getTransferEncoding()));
            InputStream in = decoded;
            String attachname = meta.get(MailMetaInfo.ATTACHNAME);
            String dispositionType = meta.get(MailMetaInfo.DISPOSITION_TYPE);
            boolean isAttach = "attachment".equals(dispositionType)
                || (attachname != null && !"inline".equals(dispositionType));
            MD5InputStream md5is = null;
            CountingInputStream cis = null;
            if (isAttach) {
                md5is = new MD5InputStream(in);
                cis = new CountingInputStream(md5is);
                in = cis;
            }
            int maxPartLength = options.maxPartLength();
            if (maxPartLength >= 0) {
                in = new LimitedInputStream(in, maxPartLength);
            }
            Throwable error = null;
            boolean malformedInput = false;
            try {
                document.process(in, bd.getCharset(), isAttach);
            } catch (IOException e) {
                try {
                    decoded.throwIfCauseOf(e);
                } catch (IOException ex) {
                    malformedInput = true;
                    e = ex;
                }
                tagged.throwIfCauseOf(e);
                error = e;
            }
            document.addError(error);
            if (isAttach) {
                if (malformedInput) {
                    document.addField(MailMetaInfo.ATTACHSIZE, 0);
                } else {
                    long size = cis.pos();
                    document.addField(MailMetaInfo.ATTACHSIZE, size);
                    if (size > 0) {
                        document.addField(
                            MailMetaInfo.MD5,
                            new String(
                                HexStrings.UPPER.process(md5is.hash())));
                    }
                }
            }
        }
    }

    protected MailDocument createDocument(final String hid) throws IOException {
        return createDocument(hid, meta);
    }

    protected MailDocument createDocument(
        final String hid,
        final MailMetaInfo meta)
        throws IOException
    {
        return collector().createDocument(meta, hid, options);
    }

    public static boolean allowHid(
        final HidCounter hidCounter,
        final TextExtractOptions options)
    {
        return hidCounter.depth() <= options.maxHidDepth()
            && hidCounter.partsCount() < options.maxParts();
    }

    protected static String hid(
        final HidCounter hidCounter,
        final TextExtractOptions options)
    {
        if (allowHid(hidCounter, options)) {
            hidCounter.incrementPartsCount();
            return hidCounter.hid();
        } else {
            return null;
        }
    }

    @Override
    @SuppressWarnings("try")
    public void body(final BodyDescriptor bd, final InputStream is)
        throws IOException
    {
        String hid = hid(meta.counter(), options);
        if (hid == null) {
            is.close();
        } else {
            extractText(hid, bd, is);
        }
    }
}

