package ru.yandex.tikaite.mimeparser;

import java.io.IOException;

import org.apache.james.mime4j.MimeException;

import ru.yandex.search.document.Document;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.tikaite.util.CommonFields;
import ru.yandex.tikaite.util.TextExtractOptions;

public class NestedMessageHandler extends NestedPartHandler {
    public static final String MESSAGE_RFC822 = "message/rfc822";
    public static final String ATTACHTYPE = "eml mime";
    public static final String EML = ".eml";

    // Separate meta without nested message headers
    @SuppressWarnings("HidingField")
    private final MailMetaInfo meta;
    private String hid;

    public NestedMessageHandler(
        final DocumentCollector collector,
        final MailMetaInfo meta,
        final TextExtractOptions options)
    {
        this(collector, meta, options, hid(meta.counter(), options));
    }

    // CSOFF: ParameterNumber
    private NestedMessageHandler(
        final DocumentCollector collector,
        final MailMetaInfo meta,
        final TextExtractOptions options,
        final String hid)
    {
        super(collector, new MailMetaInfo(createPart(meta)), options);
        this.meta = meta;
        this.hid = hid;
    }
    // CSON: ParameterNumber

    private static MailMetaInfo createPart(final MailMetaInfo meta) {
        meta.counter().createPart();
        return meta;
    }

    @Override
    public void endHeader() throws MimeException {
        super.endHeader();
        if (hid != null) {
            try (Document document = createDocument(hid, meta)) {
                hid = null;
                if (super.meta.get(MailMetaInfo.ATTACHNAME) == null) {
                    String subject = super.meta.getLocal(
                        MailMetaInfo.HDR + MailMetaInfo.SUBJECT);
                    if (subject == null) {
                        subject = EML;
                    } else {
                        subject += EML;
                    }
                    document.addField(MailMetaInfo.ATTACHNAME, subject);
                }
                // All following fields can be only written by
                // MessageHandler.processBody(...), not by MailMetaInfo.names()
                document.addField(MailMetaInfo.ATTACHSIZE, 0L);
                document.addField(MailMetaInfo.ATTACHTYPE, ATTACHTYPE);
                document.addField(CommonFields.PARSED, true);
                document.addField(CommonFields.MIMETYPE, MESSAGE_RFC822);
                document.addField(MailMetaInfo.CONTENT_TYPE, MESSAGE_RFC822);
            } catch (IOException e) {
                throw new MimeException("Failed to add nested document", e);
            }
        }
    }
}

