package ru.yandex.tikaite.parser;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipInputStream;

import org.apache.tika.exception.TikaException;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.mime.MediaType;
import org.apache.tika.parser.ParseContext;
import org.apache.tika.parser.Parser;
import org.apache.tika.sax.XHTMLContentHandler;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import ru.yandex.io.CloseShieldInputStream;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.tikaite.detect.DmarcDetector;
import ru.yandex.tikaite.detect.UngzippingDetector;
import ru.yandex.tikaite.detect.UnzippingDetector;
import ru.yandex.xml.xpath.XPathContentHandler;
import ru.yandex.xml.xpath.XPathLeafHandler;

public enum DmarcParser implements Parser {
    INSTANCE;

    private static final MediaType DMARC = DmarcDetector.DMARC;
    private static final MediaType DMARC_GZIP =
        UngzippingDetector.wrapType(DMARC);
    private static final MediaType DMARC_ZIP =
        UnzippingDetector.wrapType(DMARC);
    private static final Set<MediaType> TYPES =
        new HashSet<>(Arrays.asList(DMARC, DMARC_GZIP, DMARC_ZIP));
    private static final String DMARC_GZIP_STR = DMARC_GZIP.toString();
    private static final String DMARC_ZIP_STR = DMARC_ZIP.toString();
    private static final String P = "p";
    private static final String POLICY_PUBLISHED = "policy_published";
    private static final String POLICY_EVALUATED = "policy_evaluated";
    private static final String ORG_NAME = "org_name";
    private static final String EMAIL = "email";
    private static final String REPORT_ID = "report_id";
    private static final String DOMAIN = "domain";
    private static final String ADKIM = "adkim";
    private static final String ASPF = "aspf";
    private static final String PCT = "pct";
    private static final String HEADER_FROM = "header_from";
    private static final String SOURCE_IP = "source_ip";
    private static final String COUNT = "count";
    private static final String DISPOSITION = "disposition";
    private static final String DKIM = "dkim";
    private static final String SPF = "spf";
    private static final String RESULT = "result";

    @Override
    public Set<MediaType> getSupportedTypes(final ParseContext context) {
        return TYPES;
    }

    // CSOFF: ParameterNumber
    @Override
    public void parse(
        final InputStream is,
        final ContentHandler handler,
        final Metadata metadata,
        final ParseContext context)
        throws IOException, SAXException, TikaException
    {
        String type = metadata.get(Metadata.CONTENT_TYPE);
        XHTMLContentHandler xhtml = new XHTMLContentHandler(handler, metadata);
        xhtml.startDocument();
        xhtml.startElement(P);
        if (DMARC_GZIP_STR.equals(type)) {
            try (GZIPInputStream gzis =
                new GZIPInputStream(new CloseShieldInputStream(is)))
            {
                parse(gzis, xhtml, context);
            } catch (ZipException e) {
                throw new TikaException("Bad gzip", e);
            }
        } else if (DMARC_ZIP_STR.equals(type)) {
            try (ZipInputStream zis =
                new ZipInputStream(new CloseShieldInputStream(is)))
            {
                ZipEntry entry = zis.getNextEntry();
                if (entry == null) {
                    throw new ZipException("No zip entries found");
                } else {
                    try (ZipEntryCloser closer = new ZipEntryCloser(zis)) {
                        parse(closer.zis(), xhtml, context);
                    }
                }
            } catch (ZipException e) {
                throw new TikaException("Bad zip", e);
            }
        } else {
            parse(is, xhtml, context);
        }
        xhtml.endElement(P);
        xhtml.endDocument();
    }
    // CSON: ParameterNumber

    private static void parse(
        final InputStream in,
        final XHTMLContentHandler handler,
        final ParseContext context)
        throws IOException, SAXException, TikaException
    {
        XMLReader reader = context.getXMLReader();
        reader.setContentHandler(
            new XPathContentHandler(new DmarcHandler(handler)));
        reader.parse(new InputSource(new CloseShieldInputStream(in)));
    }

    private static class ZipEntryCloser implements Closeable {
        private final ZipInputStream zis;

        ZipEntryCloser(final ZipInputStream zis) {
            this.zis = zis;
        }

        public ZipInputStream zis() {
            return zis;
        }

        @Override
        public void close() throws IOException {
            zis.closeEntry();
        }
    }

    private static class AuthResult {
        private String domain = null;
        private String result = null;
    }

    private static class DmarcHandler implements XPathLeafHandler {
        private static final char[] EMPTY_BUF = new char[0];

        private final StringBuilder sb = new StringBuilder();
        private final JsonWriter writer =
            new JsonWriter(new StringBuilderWriter(sb));
        private final XHTMLContentHandler xhtml;
        private char[] buf = EMPTY_BUF;

        // report_metadata
        private String orgName = null;
        private String email = null;
        private String reportId = null;
        private String dateRangeBegin = null;
        private String dateRangeEnd = null;

        // policy_published
        private String domain = null;
        private String adkim = null;        // optional
        private String aspf = null;         // optional
        private String p = null;
        private String pct = null;

        // record
        private String headerFrom = null;   // record/identifiers/header_from
        private String sourceIp = null;     // record/row/source_ip
        private String count = null;        // record/row/count
        private String policyEvaluatedDisposition = null;
        private String policyEvaluatedDkim = null;
        private String policyEvaluatedSpf = null;
        private List<AuthResult> dkim = new ArrayList<>();
        private List<AuthResult> spf = new ArrayList<>();
        private AuthResult currentAuthResult = new AuthResult();

        private boolean first = true;

        DmarcHandler(final XHTMLContentHandler xhtml) {
            this.xhtml = xhtml;
        }

        @Override
        public void handle(final List<String> path, final String text)
            throws SAXException
        {
            if (path.size() >= 2 && path.get(0).equals("feedback")) {
                switch (path.get(1)) {
                    case "report_metadata":
                        handleReportMetadata(path, text);
                        break;
                    case POLICY_PUBLISHED:
                        handlePolicyPublished(path, text);
                        break;
                    case "record":
                        handleRecord(path, text);
                        break;
                    default:
                        break;
                }
            }
        }

        private void handleReportMetadata(
            final List<String> path,
            final String text)
        {
            if (path.size() == 2 + 1) {
                switch (path.get(2)) {
                    case ORG_NAME:
                        orgName = text;
                        break;
                    case EMAIL:
                        email = text;
                        break;
                    case REPORT_ID:
                        reportId = text;
                        break;
                    default:
                        break;
                }
            } else if (path.size() == 2 + 2
                && path.get(2).equals("date_range"))
            {
                switch (path.get(2 + 1)) {
                    case "begin":
                        dateRangeBegin = text;
                        break;
                    case "end":
                        dateRangeEnd = text;
                        break;
                    default:
                        break;
                }
            }
        }

        private void handlePolicyPublished(
            final List<String> path,
            final String text)
        {
            if (path.size() == 2 + 1) {
                switch (path.get(2)) {
                    case DOMAIN:
                        domain = text;
                        break;
                    case ADKIM:
                        adkim = text;
                        break;
                    case ASPF:
                        aspf = text;
                        break;
                    case P:
                        p = text;
                        break;
                    case PCT:
                        pct = text;
                        break;
                    default:
                        break;
                }
            }
        }

        private void handleRecord(
            final List<String> path,
            final String text)
            throws SAXException
        {
            if (path.size() == 2) {
                try {
                    writeRecord();
                } catch (IOException e) {
                    throw new SAXException(e);
                }
                resetRecord();
            } else {
                switch (path.get(2)) {
                    case "row":
                        handleRow(path, text);
                        break;
                    case "identifiers":
                        handleIdentifiers(path, text);
                        break;
                    case "auth_results":
                        handleAuthResults(path, text);
                        break;
                    default:
                        break;
                }
            }
        }

        private void handleRow(
            final List<String> path,
            final String text)
        {
            if (path.size() == 2 + 2) {
                switch (path.get(2 + 1)) {
                    case SOURCE_IP:
                        sourceIp = text;
                        break;
                    case COUNT:
                        count = text;
                        break;
                    default:
                        break;
                }
            } else if (path.size() == 2 + 2 + 1
                && path.get(2 + 1).equals(POLICY_EVALUATED))
            {
                switch (path.get(2 + 2)) {
                    case DISPOSITION:
                        policyEvaluatedDisposition = text;
                        break;
                    case DKIM:
                        policyEvaluatedDkim = text;
                        break;
                    case SPF:
                        policyEvaluatedSpf = text;
                        break;
                    default:
                        break;
                }
            }
        }

        private void handleIdentifiers(
            final List<String> path,
            final String text)
        {
            if (path.size() == 2 + 2
                && path.get(2 + 1).equals(HEADER_FROM))
            {
                headerFrom = text;
            }
        }

        private void handleAuthResults(
            final List<String> path,
            final String text)
        {
            if (path.size() == 2 + 2) {
                switch (path.get(2 + 1)) {
                    case DKIM:
                        dkim.add(currentAuthResult);
                        break;
                    case SPF:
                        spf.add(currentAuthResult);
                        break;
                    default:
                        break;
                }
                currentAuthResult = new AuthResult();
            } else if (path.size() == 2 + 2 + 1) {
                switch (path.get(2 + 1)) {
                    case DKIM:
                    case SPF:
                        switch (path.get(2 + 2)) {
                            case DOMAIN:
                                currentAuthResult.domain = text;
                                break;
                            case RESULT:
                                currentAuthResult.result = text;
                                break;
                            default:
                                break;
                        }
                        break;
                    default:
                        break;
                }
            }
        }

        private void writeRecord() throws IOException, SAXException {
            if (first) {
                first = false;
            } else {
                writer.reset();
                sb.setLength(0);
                sb.append('\n');
            }
            writer.startObject();
            writer.key(ORG_NAME);
            writer.value(orgName);
            writer.key(EMAIL);
            writer.value(email);
            writer.key(REPORT_ID);
            writer.value(reportId);
            writer.key("date_range_begin");
            writer.value(dateRangeBegin);
            writer.key("date_range_end");
            writer.value(dateRangeEnd);

            writer.key(POLICY_PUBLISHED);
            writer.startObject();
            writer.key(DOMAIN);
            writer.value(domain);
            writer.key(ADKIM);
            writer.value(adkim);
            writer.key(ASPF);
            writer.value(aspf);
            writer.key(P);
            writer.value(p);
            writer.key(PCT);
            writer.value(pct);
            writer.endObject();

            writer.key(HEADER_FROM);
            writer.value(headerFrom);
            writer.key(SOURCE_IP);
            writer.value(sourceIp);
            writer.key(COUNT);
            writer.value(count);

            writer.key(POLICY_EVALUATED);
            writer.startObject();
            writer.key(DISPOSITION);
            writer.value(policyEvaluatedDisposition);
            writer.key(DKIM);
            writer.value(policyEvaluatedDkim);
            writer.key(SPF);
            writer.value(policyEvaluatedSpf);
            writer.endObject();

            writer.key(DKIM);
            writer.startArray();
            for (AuthResult authResult: dkim) {
                writer.startObject();
                writer.key("dkim_domain");
                writer.value(authResult.domain);
                writer.key("dkim_result");
                writer.value(authResult.result);
                writer.endObject();
            }
            writer.endArray();

            writer.key(SPF);
            writer.startArray();
            for (AuthResult authResult: dkim) {
                writer.startObject();
                writer.key("spf_domain");
                writer.value(authResult.domain);
                writer.key("spf_result");
                writer.value(authResult.result);
                writer.endObject();
            }
            writer.endArray();

            writer.endObject();
            writer.close();

            int len = sb.length();
            if (buf.length < len) {
                buf = new char[Math.max(len, buf.length << 1)];
            }
            sb.getChars(0, len, buf, 0);
            xhtml.characters(buf, 0, len);
        }

        private void resetRecord() {
            headerFrom = null;
            sourceIp = null;
            count = null;
            policyEvaluatedDisposition = null;
            policyEvaluatedDkim = null;
            policyEvaluatedSpf = null;
            dkim.clear();
            spf.clear();
            currentAuthResult = new AuthResult();
        }
    }
}

