package ru.yandex.tikaite.parser;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.apache.tika.exception.TikaException;
import org.apache.tika.metadata.DublinCore;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.metadata.Office;
import org.apache.tika.metadata.Property;
import org.apache.tika.metadata.TikaCoreProperties;
import org.apache.tika.metadata.XMPDM;
import org.apache.tika.mime.MediaType;
import org.apache.tika.parser.ParseContext;
import org.apache.tika.parser.Parser;
import org.apache.tika.sax.TeeContentHandler;
import org.apache.tika.sax.TextContentHandler;
import org.apache.tika.sax.XHTMLContentHandler;
import org.apache.tika.sax.xpath.MatchingContentHandler;
import org.apache.tika.sax.xpath.XPathParser;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;

public enum Fb2Parser implements Parser {
    INSTANCE;

    private static final Set<MediaType> TYPESET =
        new TreeSet<>(Arrays.asList(
            MediaType.application("x-fictionbook+xml"),
            MediaType.application("x-fictionbook")));

    private static final String FB2 = "fb2";
    private static final String ID = "id";

    private static final String BOOK_TITLE = "fb2:book-title";
    private static final String AUTHOR_FIRST_NAME =
            "fb2:author/fb2:first-name";
    private static final String AUTHOR_MIDDLE_NAME =
            "fb2:author/fb2:middle-name";
    private static final String AUTHOR_LAST_NAME =
            "fb2:author/fb2:last-name";
    private static final String ANNOTATION =
            "fb2:annotation";
    private static final String GENRE =
            "fb2:genre";
    private static final String DATE =
            "fb2:date";
    private static final String KEYWORDS =
            "fb2:keywords";

    private static final String[] META_TAGS = {
        BOOK_TITLE,
        AUTHOR_FIRST_NAME,
        AUTHOR_MIDDLE_NAME,
        AUTHOR_LAST_NAME,
        ANNOTATION,
        GENRE,
        DATE,
        KEYWORDS,
    };

    private static final String DESCRIPTION_XPATH =
            "/fb2:FictionBook/fb2:description/descendant::node()";
    private static final String BODY_XPATH =
            "/fb2:FictionBook/fb2:body/descendant::node()";
    private static final String BINARY_XPATH =
            "/fb2:FictionBook/fb2:binary/node()";

    private static final String FB2_NS =
            "http://www.gribuser.ru/xml/fictionbook/2.0";

    private static final String META_XPATH_1 =
            "/fb2:FictionBook/fb2:description/"
            + "fb2:title-info/";
    private static final String META_XPATH_2 =
            "//node()";

    @Override
    public Set<MediaType> getSupportedTypes(final ParseContext parseContext) {
        return TYPESET;
    }

    private static class MetaHandler extends DefaultHandler {
        private final IdentityHashMap<String, StringBuilder> meta;
        private final String tag;

        MetaHandler(
            final IdentityHashMap<String, StringBuilder> meta,
            final String tag)
        {
            this.meta = meta;
            this.tag = tag;
        }

        @Override
        public void characters(
            final char[] ch,
            final int start,
            final int length)
        {
            StringBuilder sb = meta.get(tag);
            if (sb == null) {
                sb = new StringBuilder(length);
                meta.put(tag, sb);
            } else {
                sb.append('\n');
            }
            sb.append(ch, start, length);
        }
    }

    private static class BinaryHandler extends DefaultHandler {
        private final XHTMLContentHandler handler;

        BinaryHandler(final XHTMLContentHandler handler) {
            this.handler = handler;
        }

        //CSOFF: ParameterNumber
        @Override
        public void startElement(
            final String uri,
            final String name,
            final String qName,
            final Attributes atts)
            throws SAXException
        {
            if ("binary".equals(name)) {
                if (atts.getIndex(ID) != -1) {
                    String id = atts.getValue(ID);
                    handler.characters(id);
                    handler.newline();
                }
            }
        }
    }

    private static void addToMetadata(
        final Metadata metadata,
        final Property property,
        final StringBuilder sb)
    {
        if (sb != null) {
            metadata.add(property, sb.toString());
        }
    }

    @Override
    @SuppressWarnings("deprecation")
    public void parse(
        final InputStream inputStream,
        final ContentHandler contentHandler,
        final Metadata metadata,
        final ParseContext parseContext)
        throws IOException, SAXException, TikaException
    {
        XHTMLContentHandler xhtml =
            new XHTMLContentHandler(contentHandler, metadata);
        xhtml.startDocument();

        BinaryHandler bin = new BinaryHandler(xhtml);
        XMLReader xreader = XMLReaderFactory.createXMLReader();
        ContentHandler bodyHandler = new MatchingContentHandler(
            new TextContentHandler(xhtml),
            new XPathParser(FB2, FB2_NS).parse(BODY_XPATH));
        ContentHandler descHandler = new MatchingContentHandler(
            new TextContentHandler(xhtml),
            new XPathParser(FB2, FB2_NS).parse(DESCRIPTION_XPATH));
        ContentHandler binaryHandler = new MatchingContentHandler(
            bin,
            new XPathParser(FB2, FB2_NS).parse(BINARY_XPATH));
        List<ContentHandler> handlersList =
            new ArrayList<>();
        IdentityHashMap<String, StringBuilder> meta = new IdentityHashMap<>();
        for (String tag: META_TAGS) {
            handlersList.add(new MatchingContentHandler(
                new MetaHandler(meta, tag),
                new XPathParser(FB2, FB2_NS).parse(
                        META_XPATH_1 + tag + META_XPATH_2)));
        }
        handlersList.add(bodyHandler);
        handlersList.add(binaryHandler);
        handlersList.add(descHandler);
        ContentHandler[] handlers =
            handlersList.toArray(new ContentHandler[handlersList.size()]);
        ContentHandler handler = new TeeContentHandler(handlers);

        xreader.setContentHandler(handler);
        SAXException exception = null;
        try {
            xreader.parse(new InputSource(inputStream));
            xhtml.endDocument();
        } catch (SAXException e) {
            exception = e;
        }

        addToMetadata(
            metadata,
            TikaCoreProperties.TITLE,
            meta.get(BOOK_TITLE));
        addToMetadata(
            metadata,
            DublinCore.DESCRIPTION,
            meta.get(ANNOTATION));
        addToMetadata(metadata, XMPDM.GENRE, meta.get(GENRE));
        addToMetadata(metadata, TikaCoreProperties.PRINT_DATE, meta.get(DATE));
        addToMetadata(metadata, Office.KEYWORDS, meta.get(KEYWORDS));
        StringBuilder sb = new StringBuilder();
        for (StringBuilder part: new StringBuilder[]{
                meta.get(AUTHOR_FIRST_NAME),
                meta.get(AUTHOR_MIDDLE_NAME),
                meta.get(AUTHOR_LAST_NAME)})
        {
            if (part != null) {
                if (sb.length() > 0) {
                    sb.append(' ');
                }
                sb.append(part);
            }
        }

        if (sb.length() > 0) {
            metadata.add(TikaCoreProperties.CREATOR, sb.toString());
        }

        if (exception != null) {
            throw exception;
        }
    }
    //CSON: ParameterNumber
}

