package ru.yandex.tikaite.parser;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import com.drew.imaging.tiff.TiffProcessingException;
import com.drew.imaging.tiff.TiffReader;
import com.drew.lang.ByteArrayReader;
import com.drew.metadata.MetadataException;
import com.drew.metadata.exif.ExifTiffHandler;
import org.apache.tika.exception.TikaException;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.metadata.TIFF;
import org.apache.tika.mime.MediaType;
import org.apache.tika.parser.ParseContext;
import org.apache.tika.parser.Parser;
import org.apache.tika.parser.image.ImageMetadataExtractor;
import org.xml.sax.ContentHandler;

import ru.yandex.function.ByteArrayVoidProcessor;
import ru.yandex.function.VoidProcessor;
import ru.yandex.io.CloseShieldInputStream;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.tikaite.util.TextExtractResult;

public enum HeifParser implements Parser {
    INSTANCE;

    private static final String BAD_LOCATION = "Bad location: ";
    private static final String CANT_APPLY = ", can't apply ";

    private static final int TYPE_LENGTH = 4;
    private static final int FLAGS_LENGTH = 3;
    private static final int HALF_BYTE_SHIFT = 4;
    private static final int HALF_BYTE_MASK = 15;
    private static final long BITS = 8;

    // Construction method constants
    private static final int FILE_OFFSET = 0;
    private static final int IDAT_OFFSET = 1;
    // Unused
    // private static final int ITEM_OFFSET = 2;

    // Box types
    private static final byte[] META = "meta".getBytes(StandardCharsets.UTF_8);
    private static final byte[] ILOC = "iloc".getBytes(StandardCharsets.UTF_8);
    private static final byte[] IINF = "iinf".getBytes(StandardCharsets.UTF_8);
    private static final byte[] INFE = "infe".getBytes(StandardCharsets.UTF_8);
    private static final byte[] MIME = "mime".getBytes(StandardCharsets.UTF_8);
    private static final byte[] URI = "uri ".getBytes(StandardCharsets.UTF_8);
    private static final byte[] IPRP = "iprp".getBytes(StandardCharsets.UTF_8);
    private static final byte[] IPCO = "ipco".getBytes(StandardCharsets.UTF_8);
    private static final byte[] ISPE = "ispe".getBytes(StandardCharsets.UTF_8);
    private static final byte[] IROT = "irot".getBytes(StandardCharsets.UTF_8);
    private static final byte[] IDAT = "idat".getBytes(StandardCharsets.UTF_8);
    private static final byte[] GRID = "grid".getBytes(StandardCharsets.UTF_8);
    private static final byte[] EXIF = "Exif".getBytes(StandardCharsets.UTF_8);

    private static final MediaType HEIF = MediaType.image("heif");
    private static final Set<MediaType> TYPES = Collections.singleton(HEIF);

    @Override
    public Set<MediaType> getSupportedTypes(final ParseContext context) {
        return TYPES;
    }

    // CSOFF: ParameterNumber
    @Override
    public void parse(
        final InputStream is,
        final ContentHandler handler,
        final Metadata metadata,
        final ParseContext context)
        throws IOException, TikaException
    {
        try (DataInputStream in =
                new DataInputStream(new CloseShieldInputStream(is)))
        {
            long pos = 0;
            // Either we will hit meta or we will hit and report EOF
            while (true) {
                Box box = new Box(in);
                pos += box.size;
                if (box.data instanceof MetaBoxData) {
                    MetaBoxData meta = (MetaBoxData) box.data;
                    MetaBoxTasks tasks = new MetaBoxTasks(meta, in, pos);
                    DimensionsExtractor dimensionsExtractor =
                        new DimensionsExtractor(metadata, meta.itemProperties);
                    tasks.addExtractor(GRID, dimensionsExtractor);
                    tasks.addExtractor(EXIF, new ExifExtractor(metadata));
                    tasks.extractData();
                    dimensionsExtractor.done();
                    return;
                }
            }
        } catch (EOFException e) {
            throw new TikaException("Malformed input", e);
        }
    }
    // CSON: ParameterNumber

    private static byte[] readBytes(final DataInputStream in, final int len)
        throws IOException
    {
        byte[] bytes = new byte[len];
        in.readFully(bytes);
        return bytes;
    }

    private static byte[] readZeroTerminatedString(final DataInputStream in)
        throws IOException
    {
        try (DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream())
        {
            while (true) {
                byte b = in.readByte();
                if (b == 0) {
                    return out.toByteArray();
                }
                out.write(b);
            }
        }
    }

    private static long readValue(final DataInputStream in, final int bytes)
        throws IOException
    {
        long value = 0L;
        for (int i = 0; i < bytes; ++i) {
            int b = in.readUnsignedByte();
            value = (value << BITS) | b;
        }
        return value;
    }

    private static String toString(final byte[] buf) {
        if (buf == null) {
            return "<null>";
        } else {
            return new String(buf, StandardCharsets.UTF_8);
        }
    }

    private static void skipFully(final DataInputStream in, final long toSkip)
        throws IOException
    {
        long left = toSkip;
        while (left > 0L) {
            int n;
            if (left > Integer.MAX_VALUE) {
                n = Integer.MAX_VALUE;
            } else {
                n = (int) left;
            }
            int skipped = in.skipBytes(n);
            if (skipped <= 0) {
                throw new EOFException();
            } else {
                left -= skipped;
            }
        }
    }

    private static class Box {
        private final long size;
        private final byte[] type;
        private final BoxData data;

        @SuppressWarnings("IntLongMath")
        Box(final DataInputStream in) throws IOException {
            int size = in.readInt();
            type = readBytes(in, TYPE_LENGTH);
            long left;
            if (size == 1) {
                this.size = in.readLong();
                left = this.size - Integer.BYTES - TYPE_LENGTH - Long.BYTES;
            } else {
                this.size = size;
                left = size - Integer.BYTES - TYPE_LENGTH;
            }
            if (Arrays.equals(type, META)) {
                data = new MetaBoxData(in, left);
            } else if (Arrays.equals(type, ILOC)) {
                data = new ItemLocationBoxData(in);
            } else if (Arrays.equals(type, IINF)) {
                data = new ItemInfoBoxData(in);
            } else if (Arrays.equals(type, INFE)) {
                data = new ItemInfoEntryBoxData(in, left);
            } else if (Arrays.equals(type, IPRP)) {
                data = new ItemPropertiesBoxData(in);
            } else if (Arrays.equals(type, IPCO)) {
                data = new ItemPropertyContainerBoxData(in, left);
            } else if (Arrays.equals(type, ISPE)) {
                data = new ImageSpatialExtentsPropertyBoxData(in);
            } else if (Arrays.equals(type, IROT)) {
                data = new ImageRotationBoxData(in);
            } else if (Arrays.equals(type, IDAT)) {
                data = new ItemDataBoxData(in, left);
            } else {
                data = UnknownBoxData.INSTANCE;
            }
            left -= data.size();
            skipFully(in, left - data.size());
        }

        @Override
        public String toString() {
            return "Box[" + HeifParser.toString(type)
                + ',' + size + ',' + data + ']';
        }
    }

    private interface BoxData {
        long size();
    }

    private enum UnknownBoxData implements BoxData {
        INSTANCE;

        @Override
        public long size() {
            return 0L;
        }
    }

    private static class FullBoxData implements BoxData {
        protected final int version;
        protected final byte[] flags;

        FullBoxData(final DataInputStream in) throws IOException {
            version = in.readUnsignedByte();
            flags = readBytes(in, FLAGS_LENGTH);
        }

        @Override
        @SuppressWarnings("IntLongMath")
        public long size() {
            return Byte.BYTES + flags.length;
        }

        @Override
        public String toString() {
            return getClass().getSimpleName() + '(' + version + ',' + size()
                + ',' + Arrays.toString(flags) + ')';
        }
    }

    private static class MetaBoxData extends FullBoxData {
        private final ItemLocationBoxData itemLocation;
        private final ItemInfoBoxData itemInfo;
        private final ItemPropertiesBoxData itemProperties;
        private final ItemDataBoxData itemData;
        private final long size;

        MetaBoxData(final DataInputStream in, final long size)
            throws IOException
        {
            super(in);
            long read = super.size();
            ItemLocationBoxData itemLocation = null;
            ItemInfoBoxData itemInfo = null;
            ItemPropertiesBoxData itemProperties = null;
            ItemDataBoxData itemData = null;
            while (read < size) {
                Box box = new Box(in);
                read += box.size;
                if (box.data instanceof ItemLocationBoxData) {
                    itemLocation = (ItemLocationBoxData) box.data;
                } else if (box.data instanceof ItemInfoBoxData) {
                    itemInfo = (ItemInfoBoxData) box.data;
                } else if (box.data instanceof ItemPropertiesBoxData) {
                    itemProperties = (ItemPropertiesBoxData) box.data;
                } else if (box.data instanceof ItemDataBoxData) {
                    itemData = (ItemDataBoxData) box.data;
                }
            }
            if (read > size) {
                throw new EOFException();
            }
            this.itemLocation = itemLocation;
            this.itemInfo = itemInfo;
            this.itemProperties = itemProperties;
            this.itemData = itemData;
            this.size = read;
        }

        @Override
        public long size() {
            return size;
        }

        @Override
        public String toString() {
            return super.toString() + '(' + itemProperties
                + ',' + itemInfo + ',' + itemLocation + ')';
        }
    }

    private static class ItemLocationBoxData extends FullBoxData {
        private final int offsetSize;
        private final int lengthSize;
        private final int baseOffsetSize;
        private final int indexSize;
        private final ItemLocation[] locations;
        private final long size;

        ItemLocationBoxData(final DataInputStream in) throws IOException {
            super(in);
            int b = in.readUnsignedByte();
            long read = Byte.BYTES;
            offsetSize = b >> HALF_BYTE_SHIFT;
            lengthSize = b & HALF_BYTE_MASK;

            b = in.readUnsignedByte();
            read += Byte.BYTES;
            baseOffsetSize = b >> HALF_BYTE_SHIFT;
            if (version == 1 || version == 2) {
                indexSize = b & HALF_BYTE_MASK;
            } else {
                indexSize = 0;
            }
            int itemCount;
            if (version < 2) {
                itemCount = in.readUnsignedShort();
                read += Short.BYTES;
            } else if (version == 2) {
                itemCount = in.readInt();
                read += Integer.BYTES;
            } else {
                itemCount = 0;
            }
            locations = new ItemLocation[itemCount];
            for (int i = 0; i < itemCount; ++i) {
                int itemId;
                if (version < 2) {
                    itemId = in.readUnsignedShort();
                    read += Byte.BYTES;
                } else if (version == 2) {
                    itemId = in.readInt();
                    read += Integer.BYTES;
                } else {
                    itemId = 0;
                }

                int constructionMethod;
                if (version == 1 || version == 2) {
                    constructionMethod =
                        in.readUnsignedShort() & HALF_BYTE_MASK;
                    read += Short.BYTES;
                } else {
                    constructionMethod = FILE_OFFSET;
                }

                int dataReferenceIndex = in.readUnsignedShort();
                read += Short.BYTES;
                long baseOffset = readValue(in, baseOffsetSize);
                read += baseOffsetSize;
                int extentCount = in.readUnsignedShort();
                read += Short.BYTES;
                ItemLocation location = new ItemLocation(
                    itemId,
                    constructionMethod,
                    dataReferenceIndex,
                    baseOffset,
                    extentCount);
                for (int j = 0; j < extentCount; ++j) {
                    long extentIndex;
                    if ((version == 1 || version == 2) && indexSize > 0) {
                        extentIndex = readValue(in, indexSize);
                        read += indexSize;
                    } else {
                        extentIndex = 0L;
                    }
                    long extentOffset = readValue(in, offsetSize);
                    read += offsetSize;
                    long extentLength = readValue(in, lengthSize);
                    read += lengthSize;
                    location.extents[j] = new ItemLocationExtent(
                        extentIndex,
                        extentOffset,
                        extentLength);
                }
                locations[i] = location;
            }
            this.size = read;
        }

        @Override
        public long size() {
            return size;
        }

        @Override
        public String toString() {
            return super.toString() + '(' + offsetSize + ',' + lengthSize
                + ',' + baseOffsetSize + ',' + indexSize
                + ',' + Arrays.toString(locations) + ')';
        }
    }

    private static class ItemLocation {
        private final int itemId;
        private final int constructionMethod;
        private final int dataReferenceIndex;
        private final long baseOffset;
        private final ItemLocationExtent[] extents;

        // CSOFF: ParameterNumber
        ItemLocation(
            final int itemId,
            final int constructionMethod,
            final int dataReferenceIndex,
            final long baseOffset,
            final int extentCount)
        {
            this.itemId = itemId;
            this.constructionMethod = constructionMethod;
            this.dataReferenceIndex = dataReferenceIndex;
            this.baseOffset = baseOffset;
            extents = new ItemLocationExtent[extentCount];
        }
        // CSON: ParameterNumber

        @Override
        public String toString() {
            return "ItemLocation(" + itemId + ',' + constructionMethod
                + ',' + dataReferenceIndex + ',' + baseOffset
                + ',' + Arrays.toString(extents) + ')';
        }
    }

    private static class ItemLocationExtent {
        private final long index;
        private final long offset;
        private final long length;

        ItemLocationExtent(
            final long index,
            final long offset,
            final long length)
        {
            this.index = index;
            this.offset = offset;
            this.length = length;
        }

        @Override
        public String toString() {
            return "ItemLocationExtent(" + index
                + ',' + offset + ',' + length + ')';
        }
    }

    private static class ItemInfoBoxData extends FullBoxData {
        private final ItemInfoEntryBoxData[] entries;
        private final long size;

        ItemInfoBoxData(final DataInputStream in) throws IOException {
            super(in);
            long size = super.size();
            int entryCount;
            if (version == 0) {
                entryCount = in.readUnsignedShort();
                size += Short.BYTES;
            } else {
                entryCount = in.readInt();
                size += Integer.BYTES;
            }
            entries = new ItemInfoEntryBoxData[entryCount];
            for (int i = 0; i < entryCount; ++i) {
                Box box = new Box(in);
                size += box.size;
                if (box.data instanceof ItemInfoEntryBoxData) {
                    entries[i] = (ItemInfoEntryBoxData) box.data;
                }
            }
            this.size = size;
        }

        @Override
        public long size() {
            return size;
        }

        @Override
        public String toString() {
            return super.toString() + Arrays.toString(entries);
        }
    }

    private static class ItemInfoEntryBoxData extends FullBoxData {
        private static final byte[] EMPTY_ITEM_TYPE = new byte[0];

        private final int itemId;
        private final int itemProtectionIndex;
        private final byte[] itemType;
        private final byte[] itemName;
        private final byte[] contentType;
        private final byte[] contentEncoding;
        private final byte[] itemUriType;
        private final long size;

        ItemInfoEntryBoxData(final DataInputStream in, final long size)
            throws IOException
        {
            super(in);
            long read = super.size();
            if (version == 0 || version == 1) {
                itemId = in.readUnsignedShort();
                read += Short.BYTES;
                itemProtectionIndex = in.readUnsignedShort();
                read += Short.BYTES;
                itemType = EMPTY_ITEM_TYPE;
                itemName = readZeroTerminatedString(in);
                read += itemName.length + 1;
                contentType = readZeroTerminatedString(in);
                read += contentType.length + 1;
                if (read < size) {
                    contentEncoding = readZeroTerminatedString(in);
                    read += contentEncoding.length + 1;
                } else {
                    contentEncoding = null;
                }

                itemUriType = null;
                // For version == 1 here should be item info extension parsing
                // Don't know how to use it, so skip
            } else {
                if (version == 2) {
                    itemId = in.readUnsignedShort();
                    read += Short.BYTES;
                } else if (version == 2 + 1) {
                    itemId = in.readInt();
                    read += Integer.BYTES;
                } else {
                    itemId = 0;
                }
                itemProtectionIndex = in.readUnsignedShort();
                read += Short.BYTES;
                itemType = readBytes(in, TYPE_LENGTH);
                read += TYPE_LENGTH;
                itemName = readZeroTerminatedString(in);
                read += itemName.length + 1;
                if (Arrays.equals(itemType, MIME)) {
                    contentType = readZeroTerminatedString(in);
                    read += contentType.length + 1;
                    if (read < size) {
                        contentEncoding = readZeroTerminatedString(in);
                        read += contentEncoding.length + 1;
                    } else {
                        contentEncoding = null;
                    }
                    itemUriType = null;
                } else {
                    contentType = null;
                    contentEncoding = null;
                    if (Arrays.equals(itemType, URI)) {
                        itemUriType = readZeroTerminatedString(in);
                        read += itemUriType.length + 1;
                    } else {
                        itemUriType = null;
                    }
                }
            }
            this.size = read;
        }

        @Override
        public long size() {
            return size;
        }

        @Override
        public String toString() {
            return super.toString() + '(' + itemId + ',' + itemProtectionIndex
                + ',' + HeifParser.toString(itemType)
                + ',' + HeifParser.toString(itemName)
                + ',' + HeifParser.toString(contentType)
                + ',' + HeifParser.toString(contentEncoding)
                + ',' + HeifParser.toString(itemUriType) + ')';
        }
    }

    private static class ItemPropertiesBoxData implements BoxData {
        private final ItemPropertyContainerBoxData itemPropertyContainer;
        private final long size;

        ItemPropertiesBoxData(final DataInputStream in) throws IOException {
            Box container = new Box(in);
            if (container.data instanceof ItemPropertyContainerBoxData) {
                this.itemPropertyContainer =
                    (ItemPropertyContainerBoxData) container.data;
            } else {
                throw new EOFException(
                    "Bad container type: " + container.data);
            }
            Box association = new Box(in);
            size = container.size + association.size;
        }

        @Override
        public long size() {
            return size;
        }

        @Override
        public String toString() {
            return "ItemPropertiesBoxData(" + itemPropertyContainer + ')';
        }
    }

    private static class ItemPropertyContainerBoxData implements BoxData {
        private final ImageSpatialExtentsPropertyBoxData imageSpatialExtents;
        private final ImageRotationBoxData imageRotation;
        private final long size;

        ItemPropertyContainerBoxData(final DataInputStream in, final long size)
            throws IOException
        {
            long read = 0L;
            ImageSpatialExtentsPropertyBoxData imageSpatialExtents = null;
            ImageRotationBoxData imageRotation = null;
            while (read < size) {
                Box box = new Box(in);
                if (box.data instanceof ImageSpatialExtentsPropertyBoxData) {
                    imageSpatialExtents =
                        (ImageSpatialExtentsPropertyBoxData) box.data;
                } else if (box.data instanceof ImageRotationBoxData) {
                    imageRotation = (ImageRotationBoxData) box.data;
                }
                read += box.size;
            }
            if (read > size) {
                throw new EOFException();
            }
            this.imageSpatialExtents = imageSpatialExtents;
            this.imageRotation = imageRotation;
            this.size = read;
        }

        @Override
        public long size() {
            return size;
        }

        @Override
        public String toString() {
            return "ItemPropertyContainerBoxData(" + imageSpatialExtents
                + ',' + imageRotation + ')';
        }
    }

    private static class ImageSpatialExtentsPropertyBoxData
        extends FullBoxData
    {
        private final int width;
        private final int height;

        ImageSpatialExtentsPropertyBoxData(final DataInputStream in)
            throws IOException
        {
            super(in);
            width = in.readInt();
            height = in.readInt();
        }

        @Override
        public long size() {
            return super.size() + Integer.BYTES + Integer.BYTES;
        }

        @Override
        public String toString() {
            return super.toString() + '(' + width + ',' + height + ')';
        }
    }

    private static class ImageRotationBoxData implements BoxData {
        private static final int MASK = 3;

        private final int rotation;

        ImageRotationBoxData(final DataInputStream in) throws IOException {
            rotation = in.readUnsignedByte() & MASK;
        }

        @Override
        public long size() {
            return Byte.BYTES;
        }

        @Override
        public String toString() {
            return "ImageRotationBoxData(" + rotation + ')';
        }
    }

    private static class ItemDataBoxData implements BoxData {
        private final byte[] data;

        ItemDataBoxData(final DataInputStream in, final long size)
            throws IOException
        {
            data = readBytes(in, (int) size);
        }

        @Override
        public long size() {
            return data.length;
        }

        @Override
        public String toString() {
            return "ItemDataBox(" + Arrays.toString(data) + ')';
        }
    }

    private static class FileOffsetReadTask
        implements Comparable<FileOffsetReadTask>
    {
        private final VoidProcessor<byte[], TikaException> extractor;
        private final long offset;
        private final int length;

        FileOffsetReadTask(
            final VoidProcessor<byte[], TikaException> extractor,
            final long offset,
            final int length)
        {
            this.extractor = extractor;
            this.offset = offset;
            this.length = length;
        }

        @Override
        public int compareTo(final FileOffsetReadTask other) {
            return Long.compare(offset, other.offset);
        }

        @Override
        public int hashCode() {
            return extractor.hashCode() ^ Long.hashCode(offset) ^ length;
        }

        @Override
        public boolean equals(final Object o) {
            return this == o;
        }

        @Override
        public String toString() {
            return "FileOffsetReadTask(" + extractor + ',' + offset + ','
                + length + ')';
        }
    }

    private static class MetaBoxTasks {
        private final List<FileOffsetReadTask> tasks = new ArrayList<>();
        private final MetaBoxData meta;
        private final DataInputStream in;
        private long currentPos;

        MetaBoxTasks(
            final MetaBoxData meta,
            final DataInputStream in,
            final long currentPos)
        {
            this.meta = meta;
            this.in = in;
            this.currentPos = currentPos;
        }

        private ItemLocation locationFor(final byte[] type) {
            if (meta.itemInfo != null && meta.itemLocation != null) {
                for (ItemInfoEntryBoxData entry: meta.itemInfo.entries) {
                    if (entry != null && Arrays.equals(entry.itemType, type)) {
                        ItemLocation[] locations = meta.itemLocation.locations;
                        for (ItemLocation location: locations) {
                            if (location.itemId == entry.itemId) {
                                return location;
                            }
                        }
                    }
                }
            }
            return null;
        }

        private boolean processIdatOffset(
            final ItemLocation location,
            final VoidProcessor<byte[], TikaException> extractor)
            throws TikaException
        {
            if (meta.itemData == null) {
                return false;
            }
            ItemLocationExtent extent = location.extents[0];
            byte[] data = meta.itemData.data;
            long offset = location.baseOffset + extent.offset;
            if (offset + extent.length > data.length) {
                throw new TikaException(
                    BAD_LOCATION + location
                    + ", 'idat' block length is: " + data.length
                    + CANT_APPLY + extractor);
            }
            extractor.process(data, (int) offset, (int) extent.length);
            return true;
        }

        private void onOverlap(
            final ItemLocation location,
            final VoidProcessor<byte[], TikaException> extractor,
            final FileOffsetReadTask task)
            throws TikaException
        {
            throw new TikaException(
                BAD_LOCATION + location
                + ", can't insert task " + task
                + " into tasks list " + tasks
                + CANT_APPLY + extractor);
        }

        private boolean processFileOffset(
            final ItemLocation location,
            final VoidProcessor<byte[], TikaException> extractor)
            throws TikaException
        {
            ItemLocationExtent extent = location.extents[0];
            long offset = location.baseOffset + extent.offset;
            FileOffsetReadTask task =
                new FileOffsetReadTask(extractor, offset, (int) extent.length);
            int pos = Collections.binarySearch(tasks, task);
            if (pos >= 0) {
                onOverlap(location, extractor, task);
            } else {
                int insertPos = -(pos + 1);
                if (insertPos < tasks.size()) {
                    // Not last. Check there is no overlap with next task
                    FileOffsetReadTask nextTask = tasks.get(insertPos);
                    if (nextTask.offset < offset + extent.length) {
                        onOverlap(location, extractor, task);
                    }
                }
                if (insertPos > 0) {
                    // Not first. Check there is no overlap with previous task
                    FileOffsetReadTask prevTask = tasks.get(insertPos - 1);
                    if (prevTask.offset + prevTask.length > offset) {
                        onOverlap(location, extractor, task);
                    }
                }
                tasks.add(insertPos, task);
            }
            return true;
        }

        public boolean addExtractor(
            final byte[] type,
            final VoidProcessor<byte[], TikaException> extractor)
            throws TikaException
        {
            ItemLocation location = locationFor(type);
            boolean result = false;
            if (location != null && location.extents.length > 0) {
                if (location.constructionMethod == IDAT_OFFSET) {
                    result = processIdatOffset(location, extractor);
                } else if (location.constructionMethod == FILE_OFFSET) {
                    result = processFileOffset(location, extractor);
                }
            }
            return result;
        }

        public void extractData() throws TikaException {
            try {
                for (FileOffsetReadTask task: tasks) {
                    long toSkip = task.offset - currentPos;
                    if (toSkip >= 0) {
                        skipFully(in, toSkip);
                        currentPos += toSkip;
                        byte[] buf = readBytes(in, task.length);
                        currentPos += task.length;
                        task.extractor.process(buf);
                    }
                }
            } catch (IOException e) {
                throw new TikaException("Input error", e);
            }
        }
    }

    private static class DimensionsExtractor
        implements ByteArrayVoidProcessor<TikaException>
    {
        private final Metadata metadata;
        private boolean rotate = false;
        private int width = -1;
        private int height = -1;

        DimensionsExtractor(
            final Metadata metadata,
            final ItemPropertiesBoxData properties)
        {
            this.metadata = metadata;
            if (properties != null) {
                ItemPropertyContainerBoxData container =
                    properties.itemPropertyContainer;
                if (container != null) {
                    ImageRotationBoxData imageRotation =
                        container.imageRotation;
                    if (imageRotation != null) {
                        rotate = (imageRotation.rotation & 1) == 1;
                    }
                    ImageSpatialExtentsPropertyBoxData extents =
                        container.imageSpatialExtents;
                    if (extents != null) {
                        width = extents.width;
                        height = extents.height;
                    }
                }
            }
        }

        @Override
        public void process(final byte[] buf, final int off, final int len)
            throws TikaException
        {
            try (DataInputStream in = new DataInputStream(
                    new ByteArrayInputStream(buf, off, len)))
            {
                in.readByte(); // discard version
                boolean longSize = (in.readUnsignedByte() & 1) != 0;
                in.readByte(); // discard rows - 1
                in.readByte(); // discard columns - 1
                if (longSize) {
                    width = in.readInt();
                    height = in.readInt();
                } else {
                    width = in.readUnsignedShort();
                    height = in.readUnsignedShort();
                }
            } catch (IOException e) {
                throw new TikaException("Malformed 'idat' block", e);
            }
        }

        public void done() {
            if (width != -1) {
                metadata.set(TIFF.IMAGE_WIDTH, Integer.toString(width));
            }
            if (height != -1) {
                metadata.set(TIFF.IMAGE_LENGTH, Integer.toString(height));
            }
            if (rotate) {
                metadata.set(
                    TIFF.ORIENTATION,
                    Integer.toString(
                        TextExtractResult.MIN_ROTATED_ORIENTATION));
            }
        }
    }

    private static class ExifExtractor
        extends ImageMetadataExtractor
        implements ByteArrayVoidProcessor<TikaException>
    {
        ExifExtractor(final Metadata metadata) {
            super(metadata);
        }

        @Override
        public void process(final byte[] buf, final int off, final int len)
            throws TikaException
        {
            try (DataInputStream in = new DataInputStream(
                    new ByteArrayInputStream(buf, off, len)))
            {
                // Skip Exif\0\0
                int headerSize = in.readInt();
                int baseOffset = Integer.BYTES + headerSize;
                com.drew.metadata.Metadata metadata =
                    new com.drew.metadata.Metadata();
                new TiffReader().processTiff(
                    new ByteArrayReader(buf),
                    new ExifTiffHandler(metadata, null),
                    baseOffset);
                handle(metadata);
            } catch (IOException | MetadataException
                | TiffProcessingException e)
            {
                throw new TikaException("Malformed 'Exif' block", e);
            }
        }
    }
}

