package ru.yandex.tikaite.parser;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import org.apache.commons.compress.archivers.zip.ZipArchiveInputStream;
import org.apache.tika.detect.Detector;
import org.apache.tika.exception.TikaException;
import org.apache.tika.io.TikaInputStream;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.mime.MediaType;
import org.apache.tika.parser.ParseContext;
import org.apache.tika.parser.Parser;
import org.apache.tika.sax.XHTMLContentHandler;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.writer.JsonType;

public enum PkpassParser implements Detector, Parser {
    INSTANCE;

    private static final int UTF_8_0 = 0xEF;
    private static final int UTF_8_1 = 0xBB;
    private static final int UTF_8_2 = 0xBF;
    private static final int FF = 0xFF;
    private static final int FE = 0xFE;
    private static final int PEEK_SIZE = 12;
    private static final String P = "p";
    private static final String PASS = "pass.json";
    private static final String STRINGS = "pass.strings";
    private static final Charset UTF_32BE = Charset.forName("UTF-32BE");
    private static final Charset UTF_32LE = Charset.forName("UTF-32LE");
    private static final MediaType PKPASS =
        MediaType.application("vnd.apple.pkpass");
    private static final Set<MediaType> TYPES = Collections.singleton(PKPASS);

    private static boolean isFile(final ZipEntry entry) {
        return entry != null && !entry.isDirectory() && entry.getSize() > 0L;
    }

    @Override
    public MediaType detect(final InputStream in, final Metadata metadata)
        throws IOException
    {
        if (in instanceof TikaInputStream) {
            TikaInputStream tis = (TikaInputStream) in;
            byte[] magic = new byte[PEEK_SIZE];
            int len = IOStreamUtils.peek(tis, magic);
            if (ZipArchiveInputStream.matches(magic, len)) {
                try (ZipFile zip = new ZipFile(tis.getFile())) {
                    if (isFile(zip.getEntry(PASS))
                        && isFile(zip.getEntry("manifest.json"))
                        && isFile(zip.getEntry("signature")))
                    {
                        return PKPASS;
                    }
                } catch (ZipException e) {
                    // Looks like it is not a zip archive
                }
            }
        }
        return MediaType.OCTET_STREAM;
    }

    @Override
    public Set<MediaType> getSupportedTypes(final ParseContext context) {
        return TYPES;
    }

    private static StringBuilder readPass(final BufferedInputStream in)
        throws IOException
    {
        byte[] head = new byte[2 + 2];
        int len = IOStreamUtils.peek(in, head);
        int skip = 0;
        Charset charset = StandardCharsets.UTF_8;
        switch (head[0] & FF) {
            case 0:
                if (head[1] == 0) {
                    if ((head[2] & FF) == FE && (head[2 + 1] & FF) == FF) {
                        charset = UTF_32BE;
                        skip = head.length;
                    }
                } else {
                    charset = StandardCharsets.UTF_16BE;
                }
                break;
            case UTF_8_0:
                if ((head[1] & FF) == UTF_8_1 && (head[2] & FF) == UTF_8_2) {
                    // charset = StandardCharsets.UTF_8;
                    skip = 2 + 1;
                }
                break;
            case FE:
                if ((head[1] & FF) == FF) {
                    charset = StandardCharsets.UTF_16BE;
                    skip = 2;
                }
                break;
            case FF:
                if ((head[1] & FF) == FE) {
                    if (len == head.length
                        && head[2] == 0
                        && head[2 + 1] == 0)
                    {
                        charset = UTF_32LE;
                        skip = head.length;
                    } else {
                        charset = StandardCharsets.UTF_16LE;
                        skip = 2;
                    }
                }
                break;
            default:
                if (head[0] > 0) {
                    if (head[1] == 0) {
                        if (head[2] == 0) {
                            charset = UTF_32LE;
                        } else {
                            charset = StandardCharsets.UTF_16LE;
                        }
                    }
                    // else  charset = StandardCharsets.UTF_8;
                }
                break;
        }
        while (skip-- > 0) {
            in.read();
        }
        try (Reader reader = new InputStreamReader(
                in,
                charset.newDecoder()
                    .onMalformedInput(CodingErrorAction.REPLACE)
                    .onUnmappableCharacter(CodingErrorAction.REPLACE)))
        {
            return IOStreamUtils.consume(reader);
        }
    }

    @SuppressWarnings("JdkObsolete")
    @Override
    public void parse(
        final InputStream is,
        final ContentHandler handler,
        final Metadata metadata,
        final ParseContext context)
        throws IOException, SAXException, TikaException
    {
        StringBuilder pass = null;
        Map<String, String> passStrings = new HashMap<>();
        try (TikaInputStream tis = TikaInputStream.cast(is);
            ZipFile zip = new ZipFile(tis.getFile()))
        {
            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                if (isFile(entry)) {
                    String name = entry.getName();
                    if (PASS.equals(name)) {
                        try (BufferedInputStream passStream =
                                new BufferedInputStream(
                                    zip.getInputStream(entry)))
                        {
                            pass = readPass(passStream);
                        }
                    } else {
                        Path path = Paths.get(name);
                        String filename =
                            Objects.toString(path.getFileName(), null);
                        if (STRINGS.equals(filename)) {
                            try (Reader in =
                                    new InputStreamReader(
                                        zip.getInputStream(entry),
                                        StandardCharsets.UTF_8.newDecoder()
                                            .onMalformedInput(
                                                CodingErrorAction.REPLACE)
                                            .onUnmappableCharacter(
                                                CodingErrorAction.REPLACE)))
                            {
                                passStrings.put(
                                    JsonType.NORMAL.toString(
                                        Objects.toString(
                                            path.getParent(),
                                            "")),
                                    JsonType.NORMAL.toString(
                                        new String(
                                            IOStreamUtils.consume(in))));
                            }
                        }
                    }
                }
            }
        } catch (ZipException e) {
            throw new TikaException("Bad zip", e);
        }
        if (pass == null
            || pass.length() <= 2
            || pass.charAt(pass.length() - 1) != '}')
        {
            throw new TikaException("File pass.json not found in archive");
        }
        if (!passStrings.isEmpty()) {
            pass.setLength(pass.length() - 1);
            pass.append(",\"" + STRINGS + "\":{");
            boolean empty = true;
            for (Map.Entry<String, String> entry: passStrings.entrySet()) {
                if (empty) {
                    empty = false;
                } else {
                    pass.append(',');
                }
                pass.append(entry.getKey());
                pass.append(':');
                pass.append(entry.getValue());
            }
            pass.append('}');
            pass.append('}');
        }
        XHTMLContentHandler xhtml = new XHTMLContentHandler(handler, metadata);
        xhtml.startDocument();
        xhtml.startElement(P);
        xhtml.characters(new String(pass));
        xhtml.endElement(P);
        xhtml.endDocument();
    }
}

