package ru.yandex.tikaite.parser;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.Set;

import ar.com.hjg.pngj.PngReader;
import ar.com.hjg.pngj.PngjException;
import ar.com.hjg.pngj.PngjInputException;
import ar.com.hjg.pngj.chunks.PngMetadata;
import org.apache.jempbox.xmp.XMPMetadata;
import org.apache.jempbox.xmp.XMPSchema;
import org.apache.tika.exception.TikaException;
import org.apache.tika.metadata.DublinCore;
import org.apache.tika.metadata.Geographic;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.metadata.TIFF;
import org.apache.tika.metadata.TikaCoreProperties;
import org.apache.tika.mime.MediaType;
import org.apache.tika.parser.ParseContext;
import org.apache.tika.parser.Parser;
import org.apache.tika.parser.image.xmp.JempboxExtractor;
import org.apache.tika.sax.XHTMLContentHandler;
import org.apache.tika.utils.XMLReaderUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import ru.yandex.util.string.StringUtils;

public enum PngParser implements Parser {
    INSTANCE;

    private static final Set<MediaType> TYPES =
        Collections.singleton(MediaType.image("png"));

    private static final String[] CREATED_STRS = new String[] {
        // apart from official "Creation Time" keyword
        // there are also other common ways
        // to store creation and modification dates
        // including text fields with keywords
        // "date:create" or "create-date" etc
        "Creation Time",
        "date:create",
        "create:date",
        "date-create",
        "create-date",
    };
    private static final String[] MODIFIED_STRS = new String[] {
        "date:modify",
        "modify:date",
        "date-modify",
        "modify-date",
    };

    @Override
    public Set<MediaType> getSupportedTypes(final ParseContext parseContext) {
        return TYPES;
    }

    @Override
    public void parse(
        final InputStream inputStream,
        final ContentHandler contentHandler,
        final Metadata metadata,
        final ParseContext parseContext)
        throws IOException, SAXException, TikaException
    {
        try {
            PngReader pin = new PngReader(inputStream);
            pin.readSkippingAllRows();
            PngMetadata meta = pin.getMetadata();

            int cols = pin.imgInfo.cols;
            int rows = pin.imgInfo.rows;
            String comments = meta.getTxtForKey("Comment");
            String author = meta.getTxtForKey("Author");
            String description = meta.getTxtForKey("Description");
            String title = meta.getTxtForKey("Title");
            String created = "";
            for (String s: CREATED_STRS) {
                created = meta.getTxtForKey(s);
                if (!"".equals(created)) {
                    break;
                }
            }
            String modified = pin.getMetadata().getTimeAsString();
            if ("".equals(modified)) {
                for (String s: MODIFIED_STRS) {
                    modified = meta.getTxtForKey(s);
                    if (!"".equals(modified)) {
                        break;
                    }
                }
            }
            metadata.add(TikaCoreProperties.COMMENTS, comments);
            metadata.add(TikaCoreProperties.CREATED, created);
            metadata.add(TikaCoreProperties.MODIFIED, modified);
            metadata.add(TikaCoreProperties.TITLE, title);
            metadata.add(DublinCore.DESCRIPTION, description);
            metadata.add(TikaCoreProperties.CREATOR, author);
            metadata.add(TIFF.IMAGE_LENGTH, Integer.toString(rows));
            metadata.add(TIFF.IMAGE_WIDTH, Integer.toString(cols));

            String xmpText = meta.getTxtForKey("XML:com.adobe.xmp");
            if (xmpText != null) {
                XMPMetadata xmp = null;
                try {
                    Document dom =
                        XMLReaderUtils.buildDOM(
                            new ByteArrayInputStream(
                                StringUtils.getUtf8Bytes(xmpText)));
                    if (dom != null) {
                        xmp = new XMPMetadata(dom);
                    }
                } catch (IOException | SAXException e) {
                    // Ignore
                }
                if (xmp != null) {
                    xmp.addXMLNSMapping(
                        "http://ns.adobe.com/exif/1.0/",
                        XMPSchemaExif.class);
                    JempboxExtractor.extractDublinCore(xmp, metadata);
                    JempboxExtractor.extractXMPMM(xmp, metadata);
                    XMPSchema exif = xmp.getSchemaByClass(XMPSchemaExif.class);
                    if (exif != null) {
                        copyProperty(
                            exif,
                            "exif:GPSAltitude",
                            metadata,
                            Geographic.ALTITUDE.getName());
                        copyProperty(
                            exif,
                            "exif:GPSLatitude",
                            metadata,
                            Geographic.LATITUDE.getName());
                        copyProperty(
                            exif,
                            "exif:GPSLongitude",
                            metadata,
                            Geographic.LONGITUDE.getName());
                    }
                }
            }

            XHTMLContentHandler xhtml =
                new XHTMLContentHandler(contentHandler, metadata);
            xhtml.startDocument();
            xhtml.endDocument();
        } catch (PngjInputException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException) {
                throw (IOException) cause;
            } else {
                throw new IOException(e);
            }
        } catch (IOException | PngjException | SAXException | TikaException e) {
            throw new TikaException("Unable to parse PNG image", e);
        }
    }

    private static void copyProperty(
        final XMPSchema src,
        final String srcName,
        final Metadata dst,
        final String dstName)
    {
        String value = src.getTextProperty(srcName);
        if (value != null) {
            dst.add(dstName, value);
        }
    }

    public static class XMPSchemaExif extends XMPSchema {
        public XMPSchemaExif(final Element element, final String prefix) {
            super(element, prefix);
        }
    }
}

