package ru.yandex.tikaite.server;

import java.io.BufferedInputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.james.mime4j.MimeException;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.Field;

import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.search.document.mail.SafeMimeTokenStream;

public class HeadersHandler implements HttpRequestHandler {
    private static final byte[] SKIP_MARK =
        "\n</message>\n".getBytes(StandardCharsets.UTF_8);


    private final Server server;

    public HeadersHandler(Server server) {
        this.server = server;
    }

    @Override
    public void handle(
        HttpRequest request,
        HttpResponse response,
        HttpContext context)
        throws HttpException, IOException {


        Logger logger = (Logger) context.getAttribute(BaseHttpServer.LOGGER);

        CgiParams params = new CgiParams(request);
        String stid = params.get("stid", NonEmptyValidator.INSTANCE);
        String query = "/get/" + stid + "?raw";


        HttpEntity entity = server.sendStorageRequest(query, logger).getEntity();

        InputStream in = new BufferedInputStream(
            entity.getContent());
        in.mark(1);
        int first = in.read();
        if (first == -1) {
            throw new EOFException("Empty stream");
        }
        in.reset();
        if (first == '<') {
            IOStreamUtils.skipTo(in, SKIP_MARK);
        }
        SafeMimeTokenStream parser = new SafeMimeTokenStream();
        parser.parse(in);

        List<Map<String, String>> mailHeaders = new ArrayList<>();

        boolean insideHeaders = false;
        try {
            for (EntityState state = parser.getState();
                state != EntityState.T_END_OF_STREAM;
                state = parser.next()) {
                switch (state) {
                    case T_START_HEADER:
                        insideHeaders = true;
                        break;
                    case T_FIELD:
                        if (insideHeaders) {
                            Field field = parser.getField();
                            mailHeaders.add(
                                Map.of(field.getName(), field.getBody()));
                        }
                        break;
                    case T_END_HEADER:
                        insideHeaders = false;
                        break;
                    default:
                        break;
                }
            }
        } catch (MimeException e) {
            response.setEntity(new StringEntity(e.toString()));
        }

        Charset acceptedCharset = CharsetUtils.acceptedCharset(request);
        JsonType jsonConverter =
            JsonTypeExtractor.NORMAL.extract(new ScanningCgiParams(request));

        StringEntity resEntity = new StringEntity(
            jsonConverter.toString(Map.of("headers", mailHeaders)));
        resEntity.setContentType(ContentType.APPLICATION_JSON
            .withCharset(acceptedCharset).toString());
        resEntity.setChunked(true);

        response.setStatusCode(HttpStatus.SC_OK);
        response.setEntity(resEntity);
    }
}
