package ru.yandex.tikaite.server;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.entity.ContentType;
import org.apache.james.mime4j.MimeException;

import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.UnsupportedMediaTypeException;
import ru.yandex.io.GenericCloseableAdapter;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.SparseStringCollectorsFactory;
import ru.yandex.json.xpath.XPathContentHandler;
import ru.yandex.parser.uri.QueryParameter;
import ru.yandex.parser.uri.QueryParser;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.search.document.mail.JsonMailMetaHandler;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.document.mail.SafeMimeTokenStream;
import ru.yandex.tikaite.mimeparser.EnclosedParser;
import ru.yandex.tikaite.util.TextExtractOptions;

public class OfflineMailHandler extends OnlineMailHandler {
    private final Server server;

    public OfflineMailHandler(final Server server) {
        super(server);
        this.server = server;
    }

    public static void extractMeta(
        final MailMetaInfo meta,
        final HttpEntity entity)
        throws HttpException, IOException
    {
        ContentType contentType;
        try {
            contentType = ContentType.get(entity);
        } catch (RuntimeException e) {
            throw new BadRequestException("Failed to parse content type", e);
        }
        if (contentType == null
            || !contentType.getMimeType().equals(
                ContentType.APPLICATION_JSON.getMimeType()))
        {
            try (InputStream in = entity.getContent()) {
                SafeMimeTokenStream.parseMeta(in, meta);
            } catch (MimeException e) {
                throw new UnsupportedMediaTypeException(
                    "Failed to parse input",
                    e);
            }
        } else {
            try {
                new JsonParser(
                    new XPathContentHandler(
                        new JsonMailMetaHandler(meta),
                        SparseStringCollectorsFactory.INSTANCE.apply(-1L)))
                    .parse(CharsetUtils.content(entity));
            } catch (JsonException e) {
                throw new BadRequestException("Can't parse meta", e);
            }
        }
    }

    public static void extractMeta(
        final MailMetaInfo meta,
        final QueryParser queryParser)
        throws HttpException
    {
        for (QueryParameter param: queryParser) {
            meta.addCgi(param.name(), param.value().decode());
        }
    }

    public static void extractMeta(
        final MailMetaInfo meta,
        final HttpRequest request)
        throws HttpException, IOException
    {
        if (request.getRequestLine().getMethod().equals("POST")
            && request instanceof HttpEntityEnclosingRequest)
        {
            extractMeta(
                meta,
                ((HttpEntityEnclosingRequest) request).getEntity());
        } else {
            extractMeta(
                meta,
                new UriParser(request.getRequestLine().getUri())
                    .queryParser());
        }
    }

    @Override
    protected EnclosedParser prepareParser(
        final HttpRequest request,
        final Logger logger)
        throws HttpException
    {
        TextExtractOptions options = extractOptions(request);
        MailMetaInfo meta = new MailMetaInfo(
            options.headersLimit(),
            options.headerLimit(),
            context.yandexNets());
        try {
            extractMeta(meta, request);
        } catch (Throwable t) {
            throw Server.toHttpException(t);
        }
        logger.fine("Meta information extracted");
        HttpEntity entity =
            server.sendStorageRequest(
                new UriParser(request.getRequestLine().getUri()).rawPath()
                + "?raw",
                logger)
                .getEntity();
        try (GenericAutoCloseableHolder<
                IOException,
                GenericCloseableAdapter<InputStream>> is =
                    new GenericAutoCloseableHolder<>(
                        new GenericCloseableAdapter<>(entity.getContent())))
        {
            EnclosedParser parser =
                new EnclosedParser(meta, is.get().get(), options);
            logger.info("Main headers parsed");
            is.release();
            return parser;
        } catch (MimeException | EOFException e) {
            throw new UnsupportedMediaTypeException(
                "Failed to extract meta",
                e);
        } catch (Throwable t) {
            throw Server.toHttpException(t);
        }
    }

    @Override
    public String toString() {
        return "https://wiki.yandex-team.ru/ps/tikaite#"
            + "offlajjnovajakonvertacijapisem";
    }
}

