package ru.yandex.tikaite.server;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.MethodNotSupportedException;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.EntityTemplate;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;
import org.apache.james.mime4j.MimeException;

import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.server.sync.JsonContentProducerWriter;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.io.GenericCloseableAdapter;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.search.document.mail.OnlineMailMetaInfo;
import ru.yandex.tikaite.config.ImmutableTextExtractorConfig;
import ru.yandex.tikaite.mimeparser.EnclosedParser;
import ru.yandex.tikaite.util.TextExtractOptions;

public class OnlineMailHandler implements HttpRequestHandler {
    protected final TikaiteContext context;

    public OnlineMailHandler(final TikaiteContext context) {
        this.context = context;
    }

    protected TextExtractOptions extractOptions(final HttpRequest request)
        throws HttpException
    {
        return extractOptions(new CgiParams(request), context);
    }

    public static TextExtractOptions extractOptions(
        final CgiParams params,
        final TikaiteContext context)
        throws HttpException
    {
        ImmutableTextExtractorConfig config = context.textExtractorConfig();
        return DiskHandler.extractCommonOptions(
            new TextExtractOptions()
                .writeLimit(
                    params.getInt(
                        "limit",
                        config.mailLengthLimit()))
                .headersLimit(
                    params.getInt(
                        "headers-limit",
                        config.headersLengthLimit()))
                .headerLimit(
                    params.getInt(
                        "header-limit",
                        config.headerLengthLimit()))
                .maxHidDepth(
                    params.get(
                        "depth",
                        config.maxHidDepth(),
                        PositiveIntegerValidator.INSTANCE))
                .maxParts(
                    params.get(
                        "max-parts",
                        config.maxParts(),
                        PositiveIntegerValidator.INSTANCE))
                .maxPartLength(
                    params.getInt(
                        "max-part-length",
                        config.maxPartLength()))
                .noXurlsTypes(config.noXurlsTypes())
                .maxSanitizingLength(
                    params.getInt(
                        "max-sanitizing-length",
                        config.maxSanitizingLength())),
            params,
            context);
    }

    protected EnclosedParser prepareParser(
        final HttpRequest request,
        final Logger logger)
        throws HttpException
    {
        if (request.getRequestLine().getMethod().equals("POST")
            && request instanceof HttpEntityEnclosingRequest)
        {
            HttpEntityEnclosingRequest enclosingRequest =
                (HttpEntityEnclosingRequest) request;
            TextExtractOptions options = extractOptions(request);
            MailMetaInfo meta = new OnlineMailMetaInfo(
                options.headersLimit(),
                options.headerLimit(),
                context.yandexNets());
            try (GenericAutoCloseableHolder<
                    IOException,
                    GenericCloseableAdapter<InputStream>> is =
                        new GenericAutoCloseableHolder<>(
                            new GenericCloseableAdapter<>(
                                enclosingRequest.getEntity().getContent())))
            {
                EnclosedParser parser =
                    new EnclosedParser(meta, is.get().get(), options);
                logger.info("Mail has complete headers");
                // Prevent input closing
                enclosingRequest.setEntity(null);
                is.release();
                return parser;
            } catch (MimeException e) {
                throw new BadRequestException("Failed to extract meta", e);
            } catch (Throwable t) {
                throw Server.toHttpException(t);
            }
        } else {
            throw new MethodNotSupportedException("Mail body expected");
        }
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException
    {
        Charset acceptedCharset = CharsetUtils.acceptedCharset(request);
        Logger logger = (Logger) context.getAttribute(BaseHttpServer.LOGGER);
        EntityTemplate entity = new EntityTemplate(
            new JsonContentProducerWriter(
                new Parser(prepareParser(request, logger), logger),
                request));
        entity.setChunked(true);
        entity.setContentType(
            ContentType.APPLICATION_JSON
                .withCharset(acceptedCharset).toString());
        response.setStatusCode(HttpStatus.SC_OK);
        response.setEntity(entity);
    }

    @Override
    public String toString() {
        return "https://wiki.yandex-team.ru/ps/tikaite#"
            + "onlajjnovajakonvertacijapisem";
    }

    private static class Parser implements JsonValue {
        private final EnclosedParser parser;
        private final Logger logger;

        Parser(final EnclosedParser parser, final Logger logger) {
            this.parser = parser;
            this.logger = logger;
        }

        @Override
        public void writeValue(final JsonWriterBase out) throws IOException {
            try (EnclosedParser parser = this.parser) {
                parser.parse(out);
            } catch (MimeException e) {
                logger.log(Level.WARNING, "Parse failed", e);
            }
        }
    }
}

