package ru.yandex.tikaite.server;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.EmptyRunnable;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;
import ru.yandex.function.NullConsumer;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.GatewayTimeoutException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.InsufficientStorageException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.SynchronizedHttpContext;
import ru.yandex.http.util.UnsupportedMediaTypeException;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.sanitizer2.PageHeaderException;
import ru.yandex.sanitizer2.SanitizingHandler;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.search.document.DocumentFieldsFilter;
import ru.yandex.search.document.IdentityDocumentFieldsFilter;
import ru.yandex.tikaite.config.ImmutableFieldsFilterConfig;
import ru.yandex.tikaite.config.ImmutableTextExtractorConfig;
import ru.yandex.tikaite.config.ImmutableTikaiteConfig;
import ru.yandex.tikaite.util.FieldsFilter;
import ru.yandex.util.ip.ReloadableIpChecker;
import ru.yandex.util.storage.StorageClient;

public class Server
    extends BaseHttpServer<ImmutableTikaiteConfig>
    implements TikaiteContext
{
    private final StorageClient storageClient;
    private final HttpHost storageHost;
    private final ImmutableTextExtractorConfig textExtractorConfig;
    private final DocumentFieldsFilter fieldsFilter;
    private final SanitizingHandler sanitizer;
    private final ReloadableIpChecker yandexNets;

    public Server(final ImmutableTikaiteConfig config)
        throws IOException, JniWrapperException, PageHeaderException
    {
        super(config);
        storageClient = new StorageClient(
            ClientBuilder.createClient(
                config.storageConfig(),
                config.dnsConfig()),
            config.uriSuffix());
        closeChain.add(storageClient);
        storageHost = config.storageConfig().host();
        textExtractorConfig = config.textExtractorConfig();
        ImmutableFieldsFilterConfig fieldsFilterConfig =
            textExtractorConfig.fieldsFilterConfig();
        if (fieldsFilterConfig == null) {
            fieldsFilter = IdentityDocumentFieldsFilter.INSTANCE;
        } else {
            FieldsFilter fieldsFilter = new FieldsFilter(
                fieldsFilterConfig,
                new ThreadFactoryConfig(config.name() + "-Jni-")
                    .group(getThreadGroup())
                    .daemon(true));
            closeChain.add(
                new GenericNonThrowingCloseableAdapter<>(fieldsFilter));
            this.fieldsFilter = fieldsFilter;
        }
        ImmutableSanitizingConfig sanitizingConfig =
            textExtractorConfig.sanitizingConfig();
        if (sanitizingConfig == null) {
            sanitizer = null;
        } else {
            sanitizer = new SanitizingHandler(
                sanitizingConfig,
                NullConsumer.instance(),
                EmptyRunnable.INSTANCE);
            closeChain.add(
                new GenericNonThrowingCloseableAdapter<>(sanitizer));
        }
        yandexNets = textExtractorConfig
            .receivedChainParserConfig()
            .yandexNetsConfig()
            .createIpChecker();
        Map<String, HttpRequestHandler> handlers = new HashMap<>();
        handlers.put("disk", new DiskHandler(this));
        handlers.put("mail", new OfflineMailHandler(this));
        register(
            new Pattern<>("/get/", true),
            new NameSelectorHandler(handlers));
        register(new Pattern<>("/mail", true), new OnlineMailHandler(this));
        register(
            new Pattern<>("/text", true),
            new TextHandler(this),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/headers", true),
            new HeadersHandler(this));
    }

    @Override
    public ImmutableTextExtractorConfig textExtractorConfig() {
        return textExtractorConfig;
    }

    @Override
    public DocumentFieldsFilter fieldsFilter() {
        return fieldsFilter;
    }

    @Override
    public SanitizingHandler sanitizer() {
        return sanitizer;
    }

    @Override
    public ReloadableIpChecker yandexNets() {
        return yandexNets;
    }

    public CloseableHttpResponse sendStorageRequest(
        final String query,
        final Logger logger)
        throws HttpException
    {
        return sendStorageRequest(
            query,
            logger,
            new HttpClientContext(new SynchronizedHttpContext()));
    }

    public CloseableHttpResponse sendStorageRequest(
        final String query,
        final Logger logger,
        final HttpClientContext context)
        throws HttpException
    {
        try {
            return storageClient.sendStorageRequest(
                storageHost,
                query,
                logger,
                context);
        } catch (IOException e) {
            throw HttpExceptionConverter.toHttpException(e);
        }
    }

    public static HttpException toHttpException(final Throwable t) {
        if (t instanceof HttpException) {
            return (HttpException) t;
        } else if (t instanceof IOException) {
            if ("No space left on device".equals(t.getMessage())) {
                return new InsufficientStorageException(t);
            } else if (t instanceof SocketTimeoutException) {
                return new GatewayTimeoutException();
            } else {
                return new BadGatewayException("Failed to detect mimetype", t);
            }
        } else if (t instanceof RuntimeException) {
            return new UnsupportedMediaTypeException(t);
        } else {
            return new ServiceUnavailableException(t);
        }
    }
}

