package ru.yandex.tikaite.util;

import java.io.FilterWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.search.document.DocumentFieldsFilter;
import ru.yandex.tikaite.config.ImmutableFieldsFilterConfig;

public class FieldsFilter
    implements DocumentFieldsFilter, GenericAutoCloseable<RuntimeException>
{
    private final Set<String> filterFields;
    private final Set<String> allowedFields;
    private final JniWrapper filter;

    public FieldsFilter(
        final ImmutableFieldsFilterConfig config,
        final ThreadFactoryConfig threadFactoryConfig)
        throws JniWrapperException
    {
        filterFields = config.filterFields();
        Set<String> allowedFields = new HashSet<>(filterFields);
        allowedFields.addAll(config.bypassFields());
        this.allowedFields = CollectionCompactor.compact(allowedFields);

        filter = JniWrapper.create(config, threadFactoryConfig);
    }

    @Override
    public void close() {
        filter.close();
    }

    @Override
    public boolean fieldAllowed(final String fieldName) {
        return allowedFields.contains(fieldName);
    }

    @Override
    public String filterFieldValue(
        final String fieldName,
        final String fieldValue,
        final String options)
        throws IOException
    {
        if (filterFields.contains(fieldName)) {
            try {
                return filter.apply(fieldValue, options).process(null, 0, 0);
            } catch (JniWrapperException e) {
                throw new IOException(e);
            }
        } else {
            return fieldValue;
        }
    }

    @Override
    public Writer filterFieldValue(
        final String fieldName,
        final Writer writer,
        final String options)
        throws IOException
    {
        if (filterFields.contains(fieldName)) {
            final StringBuilderWriter sbw = new StringBuilderWriter();
            return new FilterWriter(sbw) {
                @Override
                @SuppressWarnings("try")
                public void close() throws IOException {
                    try (Writer closer = writer) {
                        String value = sbw.toString();
                        value =
                            filter.apply(value, options).process(null, 0, 0);
                        writer.write(value);
                    } catch (JniWrapperException e) {
                        throw new IOException(e);
                    }
                }
            };
        } else {
            return writer;
        }
    }
}

