package ru.yandex.tikaite.util;

import java.io.Writer;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.Set;

import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.sanitizer2.SanitizingHandler;
import ru.yandex.search.document.DocumentFieldsFilter;
import ru.yandex.search.document.IdentityDocumentFieldsFilter;
import ru.yandex.search.document.mail.SafeMimeTokenStream;
import ru.yandex.url.processor.UrlProcessor;

public class TextExtractOptions {
    public enum ParsingType {
        MEMORY,
        FILE,
        STREAM
    }

    public enum Mode {
        ULTRA_FAST {
            @Override
            public boolean allowTmpFiles() {
                return false;
            }

            @Override
            public TextExtractorContext textExtractorContext() {
                return TextExtractor.INSTANCE.ultraFastContext();
            }

            @Override
            public String cgiFlag() {
                return "ultra-fast-mode";
            }
        },
        FAST {
            @Override
            public boolean allowTmpFiles() {
                return false;
            }

            @Override
            public TextExtractorContext textExtractorContext() {
                return TextExtractor.INSTANCE.fastContext();
            }

            @Override
            public String cgiFlag() {
                return "fast-mode";
            }
        },
        SO2 {
            @Override
            public boolean allowTmpFiles() {
                return true;
            }

            @Override
            public TextExtractorContext textExtractorContext() {
                return TextExtractor.INSTANCE.so2Context();
            }

            @Override
            public String cgiFlag() {
                return "so2-mode";
            }
        },
        NORMAL {
            @Override
            public boolean allowTmpFiles() {
                return true;
            }

            @Override
            public TextExtractorContext textExtractorContext() {
                return TextExtractor.INSTANCE.slowContext();
            }

            @Override
            public String cgiFlag() {
                return "";
            }
        };

        public abstract String cgiFlag();

        public abstract boolean allowTmpFiles();

        public abstract TextExtractorContext textExtractorContext();
    }

    private int limit = -1;
    private int headersLimit = -1;
    private int headerLimit = -1;
    private int maxHidDepth = Integer.MAX_VALUE;
    private int maxParts = Integer.MAX_VALUE;
    private int maxPartLength = -1;
    private UrlProcessor processor = null;
    private ParsingType parsingType = ParsingType.FILE;
    private Mode mode = Mode.NORMAL;
    private Writer pureBodyWriter = null;
    private Set<Integer> noXurlsTypes = Collections.emptySet();
    private boolean parseDmarc = false;
    private String mimetypeHint = null;
    private SanitizingHandler sanitizer = null;
    private int maxSanitizingLength = -1;
    private Charset charsetHint = null;
    private DecodableByteArrayOutputStream htmlBody = null;
    private DocumentFieldsFilter fieldsFilter =
        IdentityDocumentFieldsFilter.INSTANCE;
    private String fieldsFilterOptions = null;
    private SafeMimeTokenStream mimeTokenStream = null;

    public TextExtractOptions() {
    }

    public TextExtractOptions(final TextExtractOptions options) {
        limit = options.limit;
        headersLimit = options.headersLimit;
        headerLimit = options.headerLimit;
        maxHidDepth = options.maxHidDepth;
        maxParts = options.maxParts;
        maxPartLength = options.maxPartLength;
        processor = options.processor;
        parsingType = options.parsingType;
        mode = options.mode;
        pureBodyWriter = options.pureBodyWriter;
        noXurlsTypes = options.noXurlsTypes;
        parseDmarc = options.parseDmarc;
        mimetypeHint = options.mimetypeHint;
        sanitizer = options.sanitizer;
        maxSanitizingLength = options.maxSanitizingLength;
        charsetHint = options.charsetHint;
        htmlBody = options.htmlBody;
        fieldsFilter = options.fieldsFilter;
        fieldsFilterOptions = options.fieldsFilterOptions;
        mimeTokenStream = options.mimeTokenStream;
    }

    public TextExtractOptions writeLimit(final int limit) {
        this.limit = limit;
        return this;
    }

    public int writeLimit() {
        return limit;
    }

    public TextExtractOptions headersLimit(final int headersLimit) {
        this.headersLimit = headersLimit;
        return this;
    }

    public int headersLimit() {
        return headersLimit;
    }

    public TextExtractOptions headerLimit(final int headerLimit) {
        this.headerLimit = headerLimit;
        return this;
    }

    public int headerLimit() {
        return headerLimit;
    }

    public TextExtractOptions maxHidDepth(final int maxHidDepth) {
        this.maxHidDepth = maxHidDepth;
        return this;
    }

    public int maxHidDepth() {
        return maxHidDepth;
    }

    public TextExtractOptions maxParts(final int maxParts) {
        this.maxParts = maxParts;
        return this;
    }

    public int maxParts() {
        return maxParts;
    }

    public TextExtractOptions maxPartLength(final int maxPartLength) {
        this.maxPartLength = maxPartLength;
        return this;
    }

    public int maxPartLength() {
        return maxPartLength;
    }

    public TextExtractOptions urlProcessor(final UrlProcessor processor) {
        this.processor = processor;
        return this;
    }

    public UrlProcessor urlProcessor() {
        return processor;
    }

    public TextExtractOptions parsingType(final ParsingType parsingType) {
        this.parsingType = parsingType;
        return this;
    }

    public ParsingType parsingType() {
        return parsingType;
    }

    public TextExtractOptions mode(final Mode mode) {
        this.mode = mode;
        return this;
    }

    public Mode mode() {
        return mode;
    }

    public TextExtractOptions pureBodyWriter(final Writer pureBodyWriter) {
        this.pureBodyWriter = pureBodyWriter;
        return this;
    }

    public Writer pureBodyWriter() {
        return pureBodyWriter;
    }

    public TextExtractOptions noXurlsTypes(final Set<Integer> noXurlsTypes) {
        this.noXurlsTypes = noXurlsTypes;
        return this;
    }

    public Set<Integer> noXurlsTypes() {
        return noXurlsTypes;
    }

    public TextExtractOptions parseDmarc(final boolean parseDmarc) {
        this.parseDmarc = parseDmarc;
        return this;
    }

    public boolean parseDmarc() {
        return parseDmarc;
    }

    public TextExtractOptions mimetypeHint(final String mimetypeHint) {
        this.mimetypeHint = mimetypeHint;
        return this;
    }

    public String mimetypeHint() {
        return mimetypeHint;
    }

    public TextExtractOptions sanitizer(final SanitizingHandler sanitizer) {
        this.sanitizer = sanitizer;
        return this;
    }

    public SanitizingHandler sanitizer() {
        return sanitizer;
    }

    public TextExtractOptions maxSanitizingLength(
        final int maxSanitizingLength)
    {
        this.maxSanitizingLength = maxSanitizingLength;
        return this;
    }

    public int maxSanitizingLength() {
        return maxSanitizingLength;
    }

    public TextExtractOptions charsetHint(final Charset charsetHint) {
        this.charsetHint = charsetHint;
        return this;
    }

    public Charset charsetHint() {
        return charsetHint;
    }

    public TextExtractOptions htmlBody(
        final DecodableByteArrayOutputStream htmlBody)
    {
        this.htmlBody = htmlBody;
        return this;
    }

    public DecodableByteArrayOutputStream htmlBody() {
        return htmlBody;
    }

    public TextExtractOptions fieldsFilter(
        final DocumentFieldsFilter fieldsFilter)
    {
        this.fieldsFilter = fieldsFilter;
        return this;
    }

    public DocumentFieldsFilter fieldsFilter() {
        return fieldsFilter;
    }

    public TextExtractOptions fieldsFilterOptions(
        final String fieldsFilterOptions)
    {
        this.fieldsFilterOptions = fieldsFilterOptions;
        return this;
    }

    public String fieldsFilterOptions() {
        return fieldsFilterOptions;
    }

    public TextExtractOptions mimeTokenStream(
        final SafeMimeTokenStream mimeTokenStream)
    {
        this.mimeTokenStream = mimeTokenStream;
        return this;
    }

    public SafeMimeTokenStream mimeTokenStream() {
        return mimeTokenStream;
    }

    public TemporaryDirectory createTemporaryResources() {
        return new TemporaryDirectory(parsingType != ParsingType.STREAM);
    }
}

