package ru.yandex.tskv;

import java.io.CharArrayReader;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;

public abstract class TskvParser<T extends TskvRecord> {
    protected enum State {
        NONE,
        PREFIX_OR_KEY,
        KEY,
        VALUE,
        BETWEEN_FIELDS,
        RECORD
    }

    private State state = State.NONE;
    private StringBuilder prefix;
    private StringBuilder key;
    private StringBuilder value;
    private boolean escape;
    private T record;

    private final TskvHandler<T> handler;
    private final TskvFormatConfig config;

    protected TskvParser(final TskvHandler<T> handler) {
        this(handler, new TskvFormatConfigBuilder());
    }

    protected TskvParser(
        final TskvHandler<T> handler,
        final TskvFormatConfig config)
    {
        this.handler = handler;
        this.config = config;

        prefix = new StringBuilder();
        key = new StringBuilder();
        value = new StringBuilder();
        reset();
    }

    public void reset() {
        prefix.setLength(0);
        key.setLength(0);
        value.setLength(0);
        escape = false;
        state = State.NONE;
        record = createRecord();
    }

    public void parse(final Reader reader) throws IOException {
        while (true) {
            try {
                parseWithException(reader);
                break;
            } catch (TskvException te) {
                if (!handler.onError(te)) {
                    break;
                }
            }
        }
    }

    protected abstract T createRecord();

    public void parseWithException(
        final String str)
        throws IOException, TskvException
    {
        parseWithException(new StringReader(str));
    }

    public void parseWithException(
        final char[] buf,
        final int offset,
        final int length)
        throws IOException, TskvException
    {
        parseWithException(new CharArrayReader(buf, offset, length));
    }

    public void parseWithException(final Reader reader)
        throws IOException, TskvException
    {
        process(reader);
        eof();
    }

    public void eof() throws TskvException {
        if (!escape && state == State.VALUE) {
            record.put(new String(key), new String(value));
            handle(record);
        } else if (state != State.NONE) {
            StringBuilder buffer = new StringBuilder(key);
            buffer.append(config.fieldSeparator());
            buffer.append(value);
            throw new TskvException(buffer, state, "Premature end of read");
        }
    }

    public void process(
        final Reader reader)
        throws IOException, TskvException
    {
        while (true) {
            int code = reader.read();
            char s = (char) code;

            if (code == -1) {
                break;
            }

            if (s == config.escapingSymbol()) {
                if (!escape) {
                    escape = true;
                    continue;
                }
            }

            switch (state) {
                case NONE:
                    if (Character.isWhitespace(s)) {
                        break;
                    }

                    state = State.PREFIX_OR_KEY;
                    prefix.append(s);
                    break;
                case PREFIX_OR_KEY:
                    if (!escape && s == config.fieldSeparator()) {
                        state = State.BETWEEN_FIELDS;
                    } else if (!escape && s == config.keyValueSeparator()) {
                        key.setLength(0);
                        key.append(prefix);
                        prefix.setLength(0);
                        state = State.VALUE;
                    } else if (!escape && s == config.recordSeparator()) {
                        state = State.NONE;
                        prefix.setLength(0);
                    } else {
                        prefix.append(s);
                    }

                    break;
                case BETWEEN_FIELDS:
                    if (escape || s != config.fieldSeparator()) {
                        key.append(s);
                        state = State.KEY;
                    } else if (s == config.recordSeparator()) {
                        state = State.RECORD;
                    }

                    break;
                case KEY:
                    if (!escape && s == config.keyValueSeparator()) {
                        state = State.VALUE;
                    } else {
                        key.append(s);

                        if (!escape && TskvFormat.keyEscape(code)) {
                            StringBuilder sb =
                                new StringBuilder(record.toString());
                            sb.append('\t');
                            sb.append(key);
                            throw new TskvException(
                                sb,
                                state,
                                "symbol not escaped: " + s + " code " + code);
                        }
                    }

                    break;
                case VALUE:
                    if (!escape && s == config.fieldSeparator()) {
                        record.put(
                            new String(key),
                            new String(value));

                        state = State.BETWEEN_FIELDS;
                        key.setLength(0);
                        value.setLength(0);
                        break;
                    } else if (!escape && s == config.recordSeparator()) {
                        record.put(
                            new String(key),
                            new String(value));

                        state = State.RECORD;
                    } else {
                        value.append(s);

                        if (!escape && TskvFormat.valueEscape(code)) {
                            throw new TskvException(
                                value,
                                state,
                                "symbol not escaped " + s);
                        }
                    }

                    break;
                default:
                    break;
            }

            escape = false;
            if (state == State.RECORD) {
                boolean continueRead = handle(record);
                reset();
                if (!continueRead) {
                    break;
                }
            }
        }
    }

    protected boolean handle(final T record) throws TskvException {
        return handler.onRecord(record);
    }
}
