package ru.yandex.tskv;

import java.io.IOException;
import java.io.Writer;

public class TskvWriter extends Writer {
    private final Writer writer;
    private final TskvFormatConfig config;
    private boolean lineStarted = false;

    public TskvWriter(
        final Writer writer,
        final TskvFormatConfig config)
    {
        this.writer = writer;
        this.config = config;
    }

    public TskvWriter(
        final Writer writer,
        final TskvFormatConfig config,
        final String prefix)
    {
        this.writer = writer;
        this.config = mutateConfig(config, prefix);
    }

    protected static TskvFormatConfig mutateConfig(
        final TskvFormatConfig config,
        final String prefix)
    {
        TskvFormatConfigBuilder builder = new TskvFormatConfigBuilder(config);
        builder.linePrefix(prefix);
        return builder;
    }

    protected Writer writer() {
        return writer;
    }

    protected TskvFormatConfig config() {
        return config;
    }

    public void endRecord() throws IOException {
        this.endRecord(true);
    }

    public void endRecord(final boolean separator) throws IOException {
        lineStarted = false;
        if (separator) {
            writer.write(config.recordSeparator());
        }
    }

    protected void startLine() throws IOException {
        String prefix = config.linePrefix();
        if (prefix != null && !prefix.isEmpty()) {
            writer.write(prefix);
            writer.write(config.fieldSeparator());
        }
    }

    public void writeRaw(
        final String key,
        final String value)
        throws IOException
    {
        writer.write(key);
        writer.append(config.keyValueSeparator());
        writer.write(value);
    }

    protected void writeEscaped(
        final String key,
        final String value)
        throws IOException
    {
        writeKey(key);
        writer.write(config.keyValueSeparator());
        writeValue(value);
    }

    protected void writeValue(final String value)
        throws IOException
    {
        for (int i = 0; i < value.length(); i++) {
            int c = value.charAt(i);
            if (TskvFormat.valueEscape(c)) {
                writer.write(config.escapingSymbol());
            }

            writer.write(value.charAt(i));
        }
    }

    protected void writeKey(final String key) throws IOException {
        for (int i = 0; i < key.length(); i++) {
            int c = key.charAt(i);
            if (TskvFormat.keyEscape(c)) {
                writer.write(config.escapingSymbol());
            }

            writer.write(key.charAt(i));
        }
    }

    public void write(final String... pairs) throws IOException {
        if (pairs.length % 2 != 0) {
            String message = "Arguments number expected to be even, got "
                + pairs.length;
            throw new IllegalArgumentException(message);
        }

        for (int i = 0; i < pairs.length; i += 2) {
            write(pairs[i], pairs[i + 1]);
        }
    }

    public void write(final String key, final String value) throws IOException {
        if (!lineStarted) {
            startLine();
            lineStarted = true;
        } else {
            writer.write(config.fieldSeparator());
        }

        if (!config.escapingEnabled()) {
            writeRaw(key, value);
            return;
        }

        writeEscaped(key, value);
    }

    @Override
    public void write(final char[] cbuf, final int off, final int len)
        throws IOException
    {
        if (!lineStarted) {
            startLine();
            lineStarted = true;
        } else {
            writer.write(config.fieldSeparator());
        }

        boolean key = true;
        for (int i = off; i < len; i++) {
            int c = cbuf[i];
            if (key && cbuf[i] == config.keyValueSeparator()) {
                key = false;
            }

            if (!config.escapingEnabled()) {
                writer.write(cbuf[i]);
                continue;
            }

            if (key && TskvFormat.keyEscape(c)) {
                writer.write(config.escapingSymbol());
            }

            if (!key && TskvFormat.valueEscape(c)) {
                writer.write(config.escapingSymbol());
            }

            writer.write(cbuf[i]);
        }
    }

    @Override
    public void flush() throws IOException {
        writer.flush();
    }

    @Override
    public void close() throws IOException {
        writer.close();
    }
}
