#include "utf8.h"

#include <mimeparser/utf8.h>

#include <boost/algorithm/string/trim.hpp>
#include <boost/iterator/counting_iterator.hpp>
#include <algorithm>
#include <map>

namespace NUtil {

using namespace mulca_mime;

namespace {

const std::locale RU_UTF8_LOCALE("ru_RU.UTF-8");

} // namespace

const std::locale& GetDefaultUtfLocale() {
    return RU_UTF8_LOCALE;
}

std::string Utf8Sanitized(const std::string& val) {
    std::string d;
    auto src = val.begin();
    auto end = val.end();
    auto dst = std::back_inserter(d);
    while (src != end) {
        src = is_valid_utf8_char(src, end) ? copy_utf8_char(src, dst) : next_utf8_char(src, end);
    }
    return d;
}

std::string Utf8Unquote(const std::string& val) {
    auto begin = val.find_first_not_of(" \t");
    auto end = val.find_last_not_of(" \t");

    if (begin != std::string::npos &&
        end != std::string::npos &&
        begin < end &&
        val[begin] == '"' &&
        val[end] == '"')
    {
        return val.substr(begin + 1, end - begin - 1);
    }

    return val;
}

namespace {

const std::map<wchar_t, wchar_t> CONTROLS{
    { L'\n', L'n' },
    { L'\r', L'r' },
    { L'\b', L'b' },
    { L'\f', L'f' },
    { L'\t', L't' },
};

} // namespace

std::string Utf8Backslash(
    const std::string& val,
    const std::wstring& toEscape,
    bool escapeControl)
{
    try {
        std::string ret;
        ret.reserve(val.size());

        auto end = val.end();
        auto dst = make_utf8_wo_iterator(std::back_inserter(ret));

        for (auto src = val.begin(); src != end; src = next_utf8_char(src, end)) {
            if (is_valid_utf8_char(src, end)) {
                wchar_t ch = utf8_char_to_wchar(src);
                std::map<wchar_t, wchar_t>::const_iterator ctrlIter;

                if (escapeControl && (ctrlIter = CONTROLS.find(ch)) != CONTROLS.end()) {
                    *dst++ = L'\\';
                    *dst++ = ctrlIter->second;
                } else if (toEscape.find(ch) != std::string::npos) {
                    *dst++ = L'\\';
                    *dst++ = ch;
                } else {
                    *dst++ = ch;
                }
            }
        }
        return ret;
    } catch (const std::exception& e) {
    }
    return val;
}

std::string Utf8Trim(const std::string& val) {
    auto range = boost::make_iterator_range(
        make_utf8_ro_iterator(val.begin(), val.end()),
        make_utf8_ro_iterator(val.end(), val.end()));
    auto result = boost::trim_copy(range, RU_UTF8_LOCALE);

    return {result.begin().base(), result.end().base()};
}

std::string Utf8ToLower(const std::string& val) {
    std::string result;

    std::transform(
        make_utf8_ro_iterator(val.begin(), val.end()),
        make_utf8_ro_iterator(val.end(), val.end()),
        make_utf8_wo_iterator(std::back_inserter(result)),
        [](wchar_t ch) { return std::tolower(ch, RU_UTF8_LOCALE); });

    return result;
}

// NB! Assuming valid utf-8 string
std::string Utf8ByteHead(const std::string& val, size_t maxCount) {
    try {
        using namespace std::placeholders;

        auto beg = val.begin();
        ptrdiff_t count = std::min<size_t>(maxCount, val.size());

        auto e1 = count
            ? next_utf8_char(beg + std::max<ptrdiff_t>(count - 4, 0), beg + count)
            : beg;
        auto e2 = count
            ? next_utf8_char(beg + std::max<ptrdiff_t>(count - 1, 0), val.end())
            : beg;

        auto end =
            std::lower_bound(
                boost::make_counting_iterator(make_utf8_ro_iterator(e1, val.end())),
                boost::make_counting_iterator(make_utf8_ro_iterator(e2, val.end())),
                count + 1,
                [beg](auto first, ptrdiff_t second) {
                    auto firstUtf8 = next_utf8_char(first.base());

                    return std::distance(beg, firstUtf8) < second;
                })->base();
        return std::string(beg, end);
    } catch (const std::exception&) {
        // ignore
    }
    return val;
}

} // namespace NUtil
