package ru.yandex.search.mail.yt.consumer;

import java.io.IOException;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.nio.BasicAsyncResponseProducerGenerator;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.search.mail.yt.consumer.scheduler.Job;
import ru.yandex.search.mail.yt.consumer.scheduler.Scheduler;
import ru.yandex.search.mail.yt.consumer.scheduler.SchedulerLock;
import ru.yandex.search.mail.yt.consumer.upload.SourceConsumerFactory;

public class SchedulerJobCompletedHandler
    implements HttpAsyncRequestHandler<HttpRequest>
{
    private final YtConsumer ytConsumer;
    private final Map<SourceConsumerFactory, Scheduler> schedulers;

    public SchedulerJobCompletedHandler(
        final YtConsumer ytConsumer,
        final Map<SourceConsumerFactory, Scheduler> schedulers)
    {
        this.ytConsumer = ytConsumer;
        this.schedulers = schedulers;
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        return new BasicAsyncRequestConsumer();
    }

    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session =
            new BasicProxySession(ytConsumer, exchange, context);
        SourceConsumerFactory consumer =
            session.params().getEnum(SourceConsumerFactory.class, "consumer");
        String jobId = session.params().getString("id");

        Scheduler scheduler = schedulers.get(consumer);
        if (scheduler == null) {
            throw new NotFoundException(
                "Source consumer not found " + consumer.name());
        }

        SchedulerLock lock = scheduler.getMaster();

        if (lock.locked()) {
            if (lock.master()) {
                session.logger().info("We are on master, completing");
                scheduler.completeJob(
                    jobId,
                    session.logger(),
                    new JobCompleteCallback(session));
            } else {
                BasicAsyncResponseProducerGenerator generator =
                    new BasicAsyncResponseProducerGenerator(
                        HttpStatus.SC_TEMPORARY_REDIRECT);
                generator.addHeader(
                    "Location",
                    lock.lockedHost().toURI() + session.uri().toString());

                session.response(generator.get());
            }
        } else {
            session.response(
                HttpStatus.SC_INTERNAL_SERVER_ERROR,
                "Failed to retrieve master for scheduler");
        }
    }

    private static final class JobCompleteCallback
        extends AbstractProxySessionCallback<Job.JobStatus>
    {
        private final JsonType jsonType;

        private JobCompleteCallback(
            final ProxySession session)
            throws HttpException
        {
            super(session);

            jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        @Override
        public void completed(final Job.JobStatus jobStatus) {
            session.logger().info("Job completed  with status " + jobStatus);
            if (jobStatus == Job.JobStatus.COMPLETING) {
                session.response(HttpStatus.SC_CONFLICT);
            } else {
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter jw = jsonType.create(sbw)) {
                    jw.startObject();
                    jw.key("status");
                    jw.value(jobStatus);
                    jw.endObject();
                } catch (IOException ioe) {
                    failed(ioe);
                    return;
                }

                StringEntity entity =
                    new StringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON);

                session.response(HttpStatus.SC_OK, entity);
            }
        }
    }
}

