package ru.yandex.search.mail.yt.consumer;

import java.io.IOException;
import java.net.MalformedURLException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.http.util.nio.BasicAsyncRequestProducer;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.string.BooleanParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.mail.yt.consumer.config.ImmutableYtConfig;
import ru.yandex.search.mail.yt.consumer.config.ImmutableYtConsumerConfig;
import ru.yandex.tskv.TskvAsyncConsumerFactory;
import ru.yandex.tskv.TskvRecord;

public class YtClient implements GenericAutoCloseable<IOException> {
    public static final String OPERATIONS_CHECK_PATH = "//sys/operations/";
    public static final String TAB_SEPARATED_VALUES
        = "text/tab-separated-values";
    public static final String ROW_COUNT_ATTRIBUTE = "row_count";
    public static final String YT_PARAMS_HEADER = "X-YT-Parameters";

    private static final String CONFIG_URI = "/yt-client";
    private static final String API = "/api/v3";
    private static final String TRANSATION_ID = "transaction_id";
    private static final String PATH = "path";
    private static final String RECURSIVE = "recursive";
    private static final String FORCE = "force";
    private static final String IGNORE_EXISTING = "ignore_existing";
    private static final String OAUTH_PREFIX = "OAuth ";
    private static final long MERGE_DEFAULT_TIMEOUT = 30000;
    private static final long MERGE_INTERVAL = 500;
    private static final long RETRY_INTERVAL = 500;
    private static final long HEAVY_HOST_UPDATE_INTERVAL = 60000;

    public enum NodeType {
        MAP_NODE,
        TABLE
    }

    public enum OperationStatus {
        NOT_FOUND,
        RUNNING,
        PENDING,
        COMPLETED,
        FAILED,
        ABORTED,
        REVIVING,
        INITIALIZING,
        PREPARING,
        MATERIALIZING,
        COMPLETING,
        ABORTING,
        FAILING
    }

    private final PrefixedLogger logger;
    private final AsyncClient client;
    private final HttpHost lightHost;
    private final String transactionId;
    private final String token;
    private final HeavyHostHolder heavyHostHolder = new HeavyHostHolder();
    private volatile boolean closed = false;

    public YtClient(
        final SharedConnectingIOReactor reactor,
        final String token,
        final ImmutableHttpHostConfig config)
    {
        this.logger =
            new PrefixedLogger(
                Logger.getAnonymousLogger(),
                "yt",
                "");
        this.token = token;
        this.client = new AsyncClient(reactor, config);
        lightHost = config.host();
        transactionId = null;
    }

    public YtClient(
        final HttpProxy<ImmutableYtConsumerConfig> proxy,
        final ImmutableYtConfig config)
    {
        this.token = config.token();
        this.logger =
            proxy.config().loggers().preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        CONFIG_URI)));

        client =
            proxy.client("YtClient", config.clientConfig())
            .adjustStater(
                proxy.config().staters(),
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        CONFIG_URI)));
        lightHost = config.cluster();
        transactionId = null;
    }

    public YtClient(final String transactionId, final YtClient ytClient) {
        token = ytClient.token;
        logger = ytClient.logger;
        client = ytClient.client;
        lightHost = ytClient.lightHost;
        this.transactionId = transactionId;
    }

    public YtClient(
        final YtClient ytClient,
        final PrefixedLogger logger)
    {
        this.token = ytClient.token;
        this.logger = ytClient.logger;
        this.client = ytClient.client;
        this.lightHost = ytClient.lightHost;
        this.transactionId = ytClient.transactionId;
    }

    public YtClient adjustLogger(final PrefixedLogger logger) {
        return new YtClient(this, logger);
    }

    public void start() {
        this.client.start();
    }

    public PrefixedLogger logger() {
        return logger;
    }

    protected HttpRequest applyAuth(final HttpRequest request) {
        request.addHeader(
            HttpHeaders.AUTHORIZATION,
            OAUTH_PREFIX + token);
        return request;
    }

    protected String internalStringResponseRequest(
        final HttpRequest request)
        throws YtException, InterruptedException
    {
        try {
            HttpResponse response =
                client.execute(
                    lightHost,
                    (h) -> new BasicAsyncRequestProducer(
                        h,
                        applyAuth(request)),
                    EmptyFutureCallback.INSTANCE).get();
            return CharsetUtils.toString(response.getEntity());
        } catch (ExecutionException | HttpException | IOException e) {
            throw new YtException(e);
        }
    }

    protected JsonObject internalJsonResponseRequest(
        final HttpRequest request)
        throws YtException, InterruptedException
    {
        try {
            return
                client.execute(
                    lightHost,
                    (h) -> new BasicAsyncRequestProducer(
                        h,
                        applyAuth(request)),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    EmptyFutureCallback.INSTANCE).get();
        } catch (ExecutionException e) {
            throw new YtException(e);
        }
    }

    protected String makeAttributes(
        final String... attributes)
        throws BadRequestException
    {
        StringBuilderWriter attrWriter =
            new StringBuilderWriter(new StringBuilder());

        try (JsonWriter writer = JsonType.NORMAL.create(attrWriter)) {
            writer.startObject();
            writer.key("attributes");
            writer.startArray();
            for (String attribute: attributes) {
                writer.value(attribute);
            }

            writer.endArray();
            writer.endObject();
        } catch (IOException e) {
            // should never happen
            throw new BadRequestException(e);
        }

        return attrWriter.toString();
    }

    public OperationStatus getOperationStatus(
        final String opId)
        throws YtException, InterruptedException
    {
        String path = OPERATIONS_CHECK_PATH + opId;
        if (!exists(path)) {
            return OperationStatus.NOT_FOUND;
        }

        String status = getAttribute(path, "state");
        return OperationStatus.valueOf(status.toUpperCase(Locale.ROOT));
    }

    public Long getLongAttribute(
        final String path,
        final String attr)
        throws YtException, InterruptedException
    {
        List<JsonObject> attrs = get(path, attr);
        try {
            if (attrs.size() <= 0) {
                throw new YtException("No attribute found");
            }

            return attrs.get(0).asLong();
        } catch (JsonException je) {
            throw new YtException(je);
        }
    }

    public String getAttribute(
        final String path,
        final String attribute)
        throws YtException, InterruptedException
    {
        List<JsonObject> attrs = get(path, attribute);
        try {
            return attrs.get(0).asStringOrNull();
        } catch (JsonException je) {
            throw new YtException(je);
        }
    }

    public List<JsonObject> get(
        final String path,
        final String... attributes)
        throws YtException, InterruptedException
    {
        QueryConstructor qc = new QueryConstructor(API + "/get?");
        HttpGet get;
        try {
            qc.append(PATH, path);
            if (transactionId != null) {
                qc.append(TRANSATION_ID, transactionId);
            }
            get = new HttpGet(qc.toString());

            if (attributes.length > 0) {
                get.addHeader(YT_PARAMS_HEADER, makeAttributes(attributes));
            }
        } catch (BadRequestException e) {
            throw new YtException(e);
        }

        JsonObject root = internalJsonResponseRequest(get);
        try {
            List<JsonObject> result = new ArrayList<>();

            JsonMap rootMap = root.asMap();
            JsonMap attrs = rootMap.getMap("$attributes");
            for (String attr: attributes) {
                result.add(attrs.get(attr));
            }

            return result;
        } catch (JsonException je) {
            throw new YtException("Invalid yt response " + root.toString(), je);
        }
    }

    public boolean exists(final String path)
        throws YtException, InterruptedException
    {
        QueryConstructor qc = new QueryConstructor(API + "/exists?");
        try {
            qc.append(PATH, path);
        } catch (BadRequestException bre) {
            throw new YtException(bre);
        }

        String result =
            internalStringResponseRequest(new HttpGet(qc.toString()));

        try {
            return BooleanParser.INSTANCE.apply(result);
        } catch (IllegalArgumentException iae) {
            throw new YtException(
                "Bad path status " + path + ' ' + result, iae);
        }
    }

    public List<String> list(
        final String path,
        final Function<String, String> resolver)
        throws YtException, InterruptedException
    {
        QueryConstructor qc = new QueryConstructor(API + "/list?");

        try {
            qc.append(PATH, path);

            JsonObject root =
                internalJsonResponseRequest(new HttpGet(qc.toString()));
            JsonList nodesObj = root.asList();
            List<String> nodes = new ArrayList<>(nodesObj.size());
            for (JsonObject node: nodesObj) {
                nodes.add(resolver.apply(node.asString()));
            }

            return nodes;
        } catch (BadRequestException | JsonException je) {
            throw new YtException(
                "Yt responsed with malformed json for " + path,
                je);
        }
    }

    public List<String> list(
        final String path)
        throws YtException, InterruptedException
    {
        return this.list(path, (s) -> s);
    }

    public List<String> listFullPathes(
        final String path)
        throws YtException, InterruptedException
    {
        return list(path, (name) -> path + '/' + name);
    }

    public HttpHost getHeavyHost() throws InterruptedException {
        return heavyHostHolder.get();
    }

    public List<TskvRecord> readTskv(
        final String path)
        throws YtException, InterruptedException
    {
        try {
            return readTskv(path, EmptyFutureCallback.INSTANCE).get();
        } catch (ExecutionException e) {
            throw new YtException("Failed to read " + path, e);
        }
    }

    public Future<List<TskvRecord>> readTskv(
        final String path,
        final FutureCallback<? super List<TskvRecord>> callback)
        throws YtException, InterruptedException
    {
        List<Header> headers =
            Collections.singletonList(
                new BasicHeader(HttpHeaders.ACCEPT, TAB_SEPARATED_VALUES));
        return read(
            path,
            headers,
            TskvAsyncConsumerFactory.OK,
            callback);
    }

    // CSOFF: ParameterNumber
    public <T> Future<T> read(
        final String path,
        final List<Header> headers,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final FutureCallback<? super T> callback)
        throws YtException, InterruptedException
    {
        HttpHost host = getHeavyHost();

        QueryConstructor qc = new QueryConstructor(API + "/read_table?");
        try {
            qc.append(PATH, path);
        } catch (BadRequestException be) {
            throw new YtException("Invalid path supplied " + path, be);
        }

        String uri = qc.toString();

        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(uri);
        headers.forEach(generator::addHeader);
        generator.addHeader(
            HttpHeaders.AUTHORIZATION,
            OAUTH_PREFIX + token);

        StringBuilder requestLog = new StringBuilder();
        requestLog.append("curl ");
        for (Header header: headers) {
            requestLog.append("-H \'");
            requestLog.append(header.toString());
            requestLog.append("\' ");
        }

        requestLog.append('\'');
        requestLog.append(host.toHostString());
        requestLog.append(uri);
        requestLog.append('\'');
        logger.info(requestLog.toString());

        return client.execute(host, generator, consumerFactory, callback);
    }
    // CSON: ParameterNumber

    public TskvRecord readTskvFirst(
        final String path)
        throws YtException,
        InterruptedException
    {
        List<TskvRecord> records = readTskv(path);
        if (records.size() > 0) {
            return records.get(0);
        }

        return null;
    }

    public void write(
        final String path,
        final Collection<TskvRecord> records)
        throws YtException, InterruptedException
    {
        HttpHost host = getHeavyHost();

        QueryConstructor qc =
            new QueryConstructor(API + "/write_table?");

        try {
            qc.append(PATH, path);

            if (transactionId != null) {
                qc.append(TRANSATION_ID, transactionId);
            }
        } catch (BadRequestException bre) {
            throw new YtException(bre);
        }

        HttpPut put = new HttpPut(qc.toString());
        put.addHeader(
            HttpHeaders.AUTHORIZATION,
            OAUTH_PREFIX + token);
        put.addHeader(HttpHeaders.CONTENT_TYPE, TAB_SEPARATED_VALUES);
        StringBuilder sb = new StringBuilder();
        if (records.size() > 0) {
            for (TskvRecord record: records) {
                sb.append(record.toString());
                sb.append('\n');
            }
        } else {
            logger.warning(qc.toString() + " empty put data");
        }

        put.setEntity(new StringEntity(sb.toString(), StandardCharsets.UTF_8));

        StringBuilder requestLog = new StringBuilder();
        requestLog.append(put.getURI().toASCIIString());
        for (Header header: put.getAllHeaders()) {
            requestLog.append(' ');
            requestLog.append(header.toString());
        }

        requestLog.append(' ');
        requestLog.append(host.toHostString());

        logger.info(requestLog.toString());
        try {
            HttpResponse response =
                client.execute(
                    host,
                    (h) -> new BasicAsyncRequestProducer(h, put),
                    EmptyFutureCallback.INSTANCE)
                    .get();
            if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
                throw new YtException(new BadResponseException(put, response));
            }
        } catch (HttpException
            | ExecutionException
            | InterruptedException
            | IOException ioe)
        {
            throw new YtException(ioe);
        }
    }

    public void write(
        final String path,
        final TskvRecord record)
        throws YtException, InterruptedException
    {
        write(path, Collections.singletonList(record));
    }

    // CSOFF: ParameterNumber
    public void move(
        final String src,
        final String dst,
        final boolean recursive,
        final boolean ignoreExist,
        final boolean force)
        throws YtException, InterruptedException
    {
        QueryConstructor qc = new QueryConstructor(API + "/move?");
        try {
            if (transactionId != null) {
                qc.append(TRANSATION_ID, transactionId);
            }
            qc.append("source_path", src);
            qc.append("destination_path", dst);
            qc.append(RECURSIVE, String.valueOf(recursive));
            qc.append(IGNORE_EXISTING, String.valueOf(ignoreExist));
            qc.append(FORCE, String.valueOf(force));
        } catch (BadRequestException bre) {
            throw new YtException(bre);
        }

        internalJsonResponseRequest(new HttpPost(qc.toString()));
    }

    // CSON: ParameterNumber

    public void move(
        final String src,
        final String dst)
        throws YtException, InterruptedException
    {
        move(src, dst, false, false, false);
    }

    public void merge(
        final String src,
        final String dst)
        throws YtException, InterruptedException
    {
        merge(Collections.singletonList(src), dst, MERGE_DEFAULT_TIMEOUT);
    }

    public void merge(
        final List<String> src,
        final String dst,
        final long timeout)
        throws YtException, InterruptedException
    {
        QueryConstructor qc = new QueryConstructor(API + "/merge?");
        try {
            if (transactionId != null) {
                qc.append(TRANSATION_ID, transactionId);
            }
        } catch (BadRequestException bre) {
            throw new YtException(bre);
        }

        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
            writer.startObject();
            writer.key("spec");
            writer.startObject();
            writer.key("input_table_paths");
            writer.startArray();
            for (String path: src) {
                writer.value(path);
            }
            writer.endArray();
            writer.key("output_table_path");
            writer.value(dst);
            writer.endObject();
            writer.endObject();
        } catch (IOException e) {
            throw new YtException(e);
        }

        HttpPost post = new HttpPost(qc.toString());
        post.setEntity(
            new StringEntity(sbw.toString(),
                ContentType.APPLICATION_JSON));
        String opId = parseStringResponse(internalStringResponseRequest(post));

        long start = System.currentTimeMillis();
        OperationStatus status = OperationStatus.NOT_FOUND;
        try {
            while (System.currentTimeMillis() - start < timeout) {
                status = getOperationStatus(opId);

                if (status == OperationStatus.FAILED
                    || status == OperationStatus.ABORTED
                    || status == OperationStatus.COMPLETED)
                {
                    break;
                }

                Thread.sleep(MERGE_INTERVAL);
                pingTransaction();
            }
        } catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
        }

        if (status != OperationStatus.COMPLETED) {
            throw new YtException(
                "Merge operation bad status " + status + ' ' + opId);
        }

        logger.info("Merge completed");
    }

    public void pingTransaction() throws YtException, InterruptedException {
        if (transactionId == null) {
            throw new YtException("Tansaction not set");
        }

        QueryConstructor qc = new QueryConstructor(API + "/ping_tx?");
        try {
            qc.append(TRANSATION_ID, transactionId);
        } catch (BadRequestException bre) {
            throw new YtException(bre);
        }

        internalStringResponseRequest(new HttpPost(qc.toString()));
    }

    public void remove(
        final String path,
        final boolean recursive,
        final boolean force)
        throws YtException, InterruptedException
    {
        QueryConstructor qc = new QueryConstructor(API + "/remove?");

        try {
            if (transactionId != null) {
                qc.append(TRANSATION_ID, transactionId);
            }
            qc.append(PATH, path);
            qc.append(RECURSIVE, String.valueOf(recursive));
            qc.append(FORCE, String.valueOf(force));
        } catch (BadRequestException bre) {
            throw new YtException(bre);
        }

        internalStringResponseRequest(new HttpPost(qc.toString()));
    }

    public void remove(final String path)
        throws YtException, InterruptedException
    {
        remove(path, false, false);
    }

    // CSOFF: ParameterNumber
    public void create(
        final String path,
        final String type,
        final boolean recursive,
        final boolean ignoreExist,
        final boolean force)
        throws YtException, InterruptedException
    {
        QueryConstructor qc = new QueryConstructor(API + "/create?");
        try {
            if (transactionId != null) {
                qc.append(TRANSATION_ID, transactionId);
            }
            qc.append(PATH, path);
            qc.append("type", type);

            qc.append(RECURSIVE, String.valueOf(recursive));
            qc.append(IGNORE_EXISTING, String.valueOf(ignoreExist));
            qc.append(FORCE, String.valueOf(force));
        } catch (BadRequestException bre) {
            throw new YtException(bre);
        }

        internalJsonResponseRequest(new HttpPost(qc.toString()));
    }
    // CSON: ParameterNumber

    public void createDirectory(
        final String path,
        final boolean recursive)
        throws YtException, InterruptedException
    {
        create(
            path,
            NodeType.MAP_NODE.name().toLowerCase(Locale.ROOT),
            recursive,
            false,
            false);
    }

    public void createTable(
        final String path,
        final boolean recursive)
        throws YtException, InterruptedException
    {
        create(
            path,
            NodeType.TABLE.name().toLowerCase(Locale.ROOT),
            recursive,
            false,
            false);
    }

    public void createTable(
        final String path)
        throws YtException, InterruptedException
    {
        createTable(path, false);
    }

    public void mkdir(
        final String path,
        final boolean recursive)
        throws BadRequestException, YtException, InterruptedException
    {
        create(
            path,
            NodeType.MAP_NODE.toString().toLowerCase(Locale.ROOT),
            recursive,
            false,
            false);
    }

    public void mkdir(
        final String path)
        throws BadRequestException, YtException, InterruptedException
    {
        mkdir(path, false);
    }

    private String parseStringResponse(final String response) {
        return response.trim().replaceAll("\"", "");
    }

    public YtClient startTransaction()
        throws YtException, InterruptedException
    {
        HttpPost post = new HttpPost(API + "/start_tx");
        String transactionId =
            parseStringResponse(internalStringResponseRequest(post));
        logger.info("Starting transaction " + transactionId);
        return new YtClient(transactionId, this);
    }

    public void abortTransaction() throws YtException, InterruptedException {
        HttpPost post = new HttpPost(
            API + "/abort_tx?" + TRANSATION_ID + '=' + transactionId);

        logger.info("Abort transaction " + transactionId);
        internalStringResponseRequest(post);
    }

    public void commitTransaction()
        throws YtException, InterruptedException
    {
        HttpPost post = new HttpPost(
            API + "/commit_tx?" + TRANSATION_ID + '=' + transactionId);

        logger.info("Commit transaction " + transactionId);
        internalStringResponseRequest(post);
    }

    public Map<String, Object> status(final boolean verbose) {
        return Collections.emptyMap();
    }

    public <T> T schedule(
        final GenericFunction<YtClient, T, ? extends Exception> func)
        throws InterruptedException
    {
        return schedule(func, RETRY_INTERVAL, logger);
    }

    public <T> T schedule(
        final GenericFunction<YtClient, T, ? extends Exception> func,
        final long retryInterval,
        final PrefixedLogger logger)
        throws InterruptedException
    {
        try {
            while (!closed) {
                try {
                    return func.apply(this);
                } catch (Exception e) {
                    logger.log(
                        Level.WARNING,
                        "Scheduled funciton  "
                            + func.getClass().getName() + " failed ",
                        e);
                }

                Thread.sleep(retryInterval);
            }
        } catch (InterruptedException ie) {
            ie.printStackTrace();
            throw ie;
        }

        return null;
    }

    @Override
    public void close() throws IOException {
        closed = true;
        client.close();
    }

    private final class HeavyHostHolder
        implements FutureCallback<JsonObject>
    {
        private final AtomicBoolean updating =
            new AtomicBoolean(false);

        private volatile HttpHost host = null;
        private volatile long ts = 0;

        private HeavyHostHolder() {
        }

        private Future<JsonObject> update(final FutureCallback<JsonObject> cb) {
            HttpGet get = new HttpGet("/hosts");
            return client.execute(
                lightHost,
                (h) -> new BasicAsyncRequestProducer(
                    h,
                    applyAuth(get)),
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                cb);
        }

        @SuppressWarnings("FutureReturnValueIgnored")
        private void updateAsync() {
            if (updating.compareAndSet(false, true)) {
                update(this);
                logger.fine("Updating heavy host");
            }
        }

        private void updateSync() throws InterruptedException {
            if (updating.compareAndSet(false, true)) {
                try {
                    update(this).get();
                } catch (ExecutionException ee) {
                    logger.log(
                        Level.WARNING,
                        "Heavy host update failed",
                        ee);
                }

                logger.fine("Updating heavy host synchronously");
            }
        }

        public HttpHost get() throws InterruptedException {
            while (host == null) {
                updateSync();
            }

            if (System.currentTimeMillis() - ts > HEAVY_HOST_UPDATE_INTERVAL) {
                updateAsync();
            }

            return host;
        }

        @Override
        public void completed(final JsonObject root) {
            HttpHost host = null;
            try {
                JsonList hosts = root.asList();
                Iterator<JsonObject> iterator = hosts.iterator();
                if (!iterator.hasNext()) {
                    failed(new Exception("No heavy host found"));
                } else {
                    JsonObject heavyProxyObj = iterator.next();
                    if (heavyProxyObj == null) {
                        failed(
                            new Exception(
                                "Bad hosts response "
                                    + JsonType.NORMAL.toString(root)));
                    } else {
                        host =
                            HttpHostParser.INSTANCE.apply(
                                heavyProxyObj.asString());
                    }
                }
            } catch (JsonException | MalformedURLException e) {
                failed(e);
            } finally {
                updating.set(false);
            }

            if (host == null) {
                updateAsync();
            } else {
                this.ts = System.currentTimeMillis();
                this.host = host;
                logger.fine("Heavy host updated to " + host.toString());
            }
        }

        @Override
        public void failed(final Exception e) {
            updating.set(false);
            updateAsync();
            logger.log(
                Level.WARNING,
                "Heavy host request failed",
                e);
        }

        @Override
        public void cancelled() {
            logger.warning("Heavy host request cancelled");
            updating.set(false);
        }
    }
}
