package ru.yandex.search.mail.yt.consumer;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpHost;

import ru.yandex.collection.Pattern;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.mail.yt.consumer.config.ImmutableYtConsumerConfig;
import ru.yandex.search.mail.yt.consumer.config.SourceConsumerBuildContext;
import ru.yandex.search.mail.yt.consumer.scheduler.Scheduler;
import ru.yandex.search.mail.yt.consumer.upload.SourceConsumerFactory;
import ru.yandex.search.mail.yt.consumer.upload.UploadWorkersManager;
import ru.yandex.search.mail.yt.consumer.upload.UploaderFactory;

public class YtConsumer extends HttpProxy<ImmutableYtConsumerConfig> {
    private final YtClient yt;
    private final UploadWorkersManager workersManager;
    private final Map<SourceConsumerFactory, SourceConsumer> consumers;
    private final AsyncClient schedulersClient;
    private final HttpHost currentHost;

    public YtConsumer(
        final ImmutableYtConsumerConfig config)
        throws IOException
    {
        super(config);

        this.schedulersClient =
            client("SchedulersClient", config.schedulerClientConfig());
        this.currentHost = detectCurrentHost();
        logger().info("Current YtConsumer host is " + currentHost);

        yt = new YtClient(this, config.ytConfig());

        consumers = new LinkedHashMap<>();
        Map<SourceConsumerFactory, Scheduler> schedulers
            = new LinkedHashMap<>();
        this.workersManager =
            new UploadWorkersManager(config, this);
        try {
            for (SourceConsumerBuildContext context : config.consumers()) {
                SourceConsumer consumer = context.build(this);
                logger().info("Found consumer " + context.factory().name());
                consumers.put(context.factory(), consumer);
                schedulers.put(context.factory(), consumer.scheduler());
            }
        } catch (ConfigException e) {
            throw new IOException(e);
        }

        this.register(
            new Pattern<>("/isMaster", false),
            new SchedulerMasterCheckHandler(this, schedulers));
        this.register(
            new Pattern<>("/jobCompleted", false),
            new SchedulerJobCompletedHandler(this, schedulers));
        this.register(
            new Pattern<>("/jobs", false),
            new JobListHandler(this));
    }

    public AsyncClient schedulersClient() {
        return schedulersClient;
    }

    public YtClient yt() {
        return yt;
    }

    @Override
    public void start() throws IOException {
        super.start();

        consumers.values().forEach(SourceConsumer::start);
        this.workersManager.start();
    }

    @Override
    public void close() throws IOException {
        super.close();

        this.workersManager.closeGracefully();
        for (SourceConsumer consumer: consumers.values()) {
            consumer.close();
        }
    }

    public Set<HttpHost> schedulers() {
        return config.schedulers();
    }

    public HttpHost currentHost() {
        return this.currentHost;
    }

    private HttpHost detectCurrentHost() throws IOException {
        String host = System.getProperty("BSCONFIG_IHOST");
        String port = System.getProperty("BSCONFIG_IPORT");

        if (host != null && port != null) {
            try {
                return
                    HttpHostParser.INSTANCE.apply(
                        "" + host + ".search.yandex.net:" + port);
            } catch (MalformedURLException mue) {
                throw new IOException(mue);
            }
        }

        int portInt = this.port();
        if (port != null) {
            try {
                portInt = Integer.parseInt(port);
            } catch (NumberFormatException nfe) {
                throw new IOException("Bad port supplied", nfe);
            }
        }

        return new HttpHost("localhost", portInt);
    }

    public String getHostId() {
        return currentHost().toHostString();
    }

    public UploaderFactory uploader(final SourceConsumerFactory factory) {
        return consumers.get(factory).uploaderFactory();
    }

    public Scheduler scheduler(final SourceConsumerFactory factory) {
        return consumers.get(factory).scheduler();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> result = super.status(verbose);
        consumers.values().forEach((s) -> result.putAll(s.status(verbose)));
        result.put("UploadWorkers", workersManager.status(verbose));
        return result;
    }

    public Map<SourceConsumerFactory, SourceConsumer> consumers() {
        return consumers;
    }
}
