package ru.yandex.search.mail.yt.consumer.alice;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Level;

import org.joda.time.DateTime;

import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.mail.yt.consumer.QueueData;
import ru.yandex.search.mail.yt.consumer.config.ImmutableAliceConfig;
import ru.yandex.search.mail.yt.consumer.upload.AbstractUploader;
import ru.yandex.search.mail.yt.consumer.upload.BasicJobContext;
import ru.yandex.search.mail.yt.consumer.upload.BasicQueueData;
import ru.yandex.search.mail.yt.consumer.upload.JsonReadingYtClient;
import ru.yandex.search.mail.yt.consumer.upload.QueueAcceptor;
import ru.yandex.search.mail.yt.consumer.upload.Uploader;
import ru.yandex.search.mail.yt.consumer.upload.UploaderFactory;
import ru.yandex.tskv.TskvRecord;

public class AliceUploaderFactory implements UploaderFactory {
    private static final double THOUSAND = 1000.0;

    private static final String ALICE_PREFIX = "алис";
    private static final String YTCONSUMER = "ytconsumer";
    private static final String SERVER_TIME = "server_time";
    private static final String UUID = "uuid";
    private static final String FORM_NAME = "form_name";

    private static final String FORM_POSTFIX = ".userinfo_name";
    private static final String PREFIX = "prefix";
    private static final int NUM_SHARDS = 65534;

    private final AliceSourceConsumer consumer;
    private final ImmutableAliceConfig config;
    private final AliceDislayNameExtractor extractor;

    public AliceUploaderFactory(
        final AliceSourceConsumer consumer)
        throws ConfigException
    {
        this.consumer = consumer;
        this.config = consumer.config();

        this.extractor = new AliceDislayNameExtractor(config);
    }

    @Override
    public Uploader create(final BasicJobContext context) {
        return new AliceUploader(context);
    }

    private final class AliceUploader extends AbstractUploader<JsonObject> {
        private final String uriBase;
        private final JsonReadingYtClient yt;

        private AliceUploader(final BasicJobContext context) {
            super(consumer, context);

            this.yt = new JsonReadingYtClient(consumer.yt(), this);

            StringBuilder sb = new StringBuilder("/notify?service=");
            sb.append(config.service());
            sb.append("&source=");
            sb.append(YTCONSUMER);
            sb.append("&path=");
            sb.append(context.source());
            sb.append("&shard=");
            this.uriBase = sb.toString();
        }

        @Override
        public JsonReadingYtClient yt() {
            return yt;
        }

        @Override
        protected void parse(
            final JsonObject record,
            final QueueAcceptor acceptor)
        {
            try {
                acceptor.add(parseWithException(record.asMap()));
            } catch (JsonException je) {
                context.logger().log(
                    Level.WARNING,
                    "Failed to parse " + JsonType.NORMAL.toString(record),
                    je);
            }
        }

        private String parseUtterance(final String text) {
            List<String> found = extractor.parse(text);
            String result = null;
            if (found.size() > 0) {
                Optional<String> first =
                    found.stream().filter(
                        (s) -> !s.toLowerCase(Locale.ROOT)
                            .startsWith(ALICE_PREFIX))
                        .findFirst();
                result = first.orElse(found.get(0));
            }

            return result;
        }

        // CSOFF: ReturnCount
        public QueueData parseWithException(
            final JsonMap record)
            throws JsonException
        {
            String uuid = record.get(UUID).asString();
            long serverTime = record.getLong(SERVER_TIME, -1L);
            String formName = record.get(FORM_NAME).asStringOrNull();

            consumer.aliceTotal().accept(1);
            if (formName == null
                || !formName.toLowerCase(Locale.ROOT).endsWith(FORM_POSTFIX))
            {
                return null;
            }

            Double optime = serverTime * THOUSAND;
            String dataJson = null;
            StringBuilderWriter dataWriter = new StringBuilderWriter();
            try (JsonWriter writer = new JsonWriter(dataWriter)) {
                writer.startObject();
                writer.key(PREFIX);
                writer.value(uuid);
                for (Map.Entry<String, JsonObject> entry: record.entrySet()) {
                    writer.key(entry.getKey());
                    writer.value(entry.getValue());
                }
                writer.endObject();
            } catch (IOException e) {
                dataWriter = null;
                context.logger().log(
                    Level.WARNING,
                    "Bad record json data",
                    e);
            }

            if (dataWriter != null) {
                dataJson = dataWriter.toString();
            }

            if (dataJson == null || uuid == null) {
                context.logger().info(
                    "Data or optime or uuid null, skipping "
                        + record.toString());

                return null;
            }

            int shard = uuid.hashCode() % NUM_SHARDS;
            if (shard < 0) {
                shard = NUM_SHARDS - shard;
            }

            String uri = uriBase + shard + "&uuid=" + uuid;

            String utteranceText =
                record.getString("utterance_text", null);
            String parsed = null;
            if (utteranceText != null) {
                parsed = parseUtterance(utteranceText);
                consumer.aliceTotalDisplayForm().accept(1);
                if (parsed != null) {
                    consumer.aliceDisplayFormParsed().accept(1);
                }
            }

            return new AliceQueueData(
                shard,
                optime,
                uri,
                dataJson,
                config.service(),
                utteranceText,
                parsed);
        }
        // CSON: ReturnCount

        @Override
        protected void push() {
            List<QueueData> data = acceptor.data();
            final List<TskvRecord> toYt = new ArrayList<>(data.size());
            for (QueueData queueData: data) {
                if (queueData instanceof AliceQueueData) {
                    AliceQueueData aliceData = (AliceQueueData) queueData;
                    if (aliceData.utterance() != null) {
                        TskvRecord record = new TskvRecord();
                        record.put("input_text", aliceData.utterance());
                        record.put(
                            "parsed",
                            String.valueOf(aliceData.parsed()));
                        toYt.add(record);
                    }
                }
            }

            super.push();

            DateTime dt = new DateTime();
            String name =
                dt.dayOfMonth().getAsString()
                    + '-' + dt.monthOfYear().getAsString()
                    + '-' + dt.year().getAsString();
            String path = "//home/mail-search/voice/" + name;
            try {
                consumer.yt().schedule((yt) -> {
                    if (!consumer.yt().exists(path)) {
                        consumer.yt().createTable(path, true);
                    }

                    consumer.yt().write("<append=true>" + path, toYt);
                    return null;
                });
            } catch (InterruptedException ie) {
                context.logger().log(
                    Level.WARNING,
                    "Push interrupted",
                    ie);
                this.close();
            }
        }

        @Override
        protected int batchSize() {
            return config.batchSize();
        }

        @Override
        protected int failureSendDelay() {
            return config.producerPushInterval();
        }
    }

    private static final class AliceQueueData extends BasicQueueData {
        private final String utterance;
        private final String parsed;

        // CSOFF: ParameterNumber
        private AliceQueueData(
            final int shard,
            final double operationDate,
            final String uri,
            final String body,
            final String service,
            final String utterance,
            final String parsed)
        {
            super(shard, operationDate, uri, body, service);

            this.utterance = utterance;
            this.parsed = parsed;
        }
        // CSON: ParameterNumber

        private String utterance() {
            return utterance;
        }

        private String parsed() {
            return parsed;
        }
    }
}
