package ru.yandex.search.mail.yt.consumer.mobile.actions;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpHost;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.mail.yt.consumer.SourceConsumer;
import ru.yandex.search.mail.yt.consumer.YtClient;
import ru.yandex.search.mail.yt.consumer.YtConsumer;
import ru.yandex.search.mail.yt.consumer.config.ImmutableSourceConsumerConfig;
import ru.yandex.search.mail.yt.consumer.config.ImmutableWorkersConfig;
import ru.yandex.search.mail.yt.consumer.scheduler.Scheduler;
import ru.yandex.search.mail.yt.consumer.upload.UploaderFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class MobileActionsSourceConsumer implements SourceConsumer {
    private final ImmutableSourceConsumerConfig config;
    private final AsyncClient producer;
    private final List<Stater> staters;
    private final Scheduler scheduler;
    private final MobileActionsUploaderFactory uploaderFactory;
    private final YtConsumer ytConsumer;
    private final PrefixedLogger logger;

    private final TimeFrameQueue<Integer> mobileParseErrors;
    private final TimeFrameQueue<Integer> mobileTotal;

    public MobileActionsSourceConsumer(
        final YtConsumer consumer,
        final ImmutableSourceConsumerConfig config)
        throws ConfigException
    {
        this.config = config;

        this.logger =
            consumer.config().loggers().preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        "/scheduler")))
                .addPrefix("MobileActions");

        this.producer =
            consumer.client("MobileActionsProducer", this.config.producer());
        this.staters = new ArrayList<>();

        this.ytConsumer = consumer;
        this.scheduler = new MobileActionsScheduler(this);
        this.uploaderFactory = new MobileActionsUploaderFactory(this);

        mobileTotal =
            new TimeFrameQueue<>(consumer.config().metricsTimeFrame());
        consumer.registerStater(
            new PassiveStaterAdapter<>(
                mobileTotal,
                new NamedStatsAggregatorFactory<>(
                    "mobile-actions-consumer-records_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        mobileParseErrors =
            new TimeFrameQueue<>(consumer.config().metricsTimeFrame());
        consumer.registerStater(
            new PassiveStaterAdapter<>(
                mobileParseErrors,
                new NamedStatsAggregatorFactory<>(
                    "mobile-actions-consumer-errors_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    @Override
    public void start() {
        producer.start();
        scheduler.start();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        return scheduler().status(verbose);
    }

    @Override
    public Scheduler scheduler() {
        return scheduler;
    }

    @Override
    public UploaderFactory uploaderFactory() {
        return uploaderFactory;
    }

    @Override
    public void close() throws IOException {
        producer.close();
        scheduler().close();
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        synchronized (staters) {
            for (Stater stater: staters) {
                stater.stats(statsConsumer);
            }
        }
    }

    @Override
    public ImmutableSourceConsumerConfig config() {
        return config;
    }

    @Override
    public AsyncClient producer() {
        return producer;
    }

    @Override
    public AsyncClient schedulersClient() {
        return ytConsumer.schedulersClient();
    }

    @Override
    public YtClient yt() {
        return ytConsumer.yt();
    }

    @Override
    public PrefixedLogger logger() {
        return logger;
    }

    @Override
    public ImmutableWorkersConfig workersConfig() {
        return ytConsumer.config().workersConfig();
    }

    @Override
    public HttpHost currentHost() {
        return ytConsumer.currentHost();
    }

    public TimeFrameQueue<Integer> mobileParseErrors() {
        return mobileParseErrors;
    }

    public TimeFrameQueue<Integer> mobileTotal() {
        return mobileTotal;
    }
}
