package ru.yandex.search.mail.yt.consumer.scheduler;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.concurrent.BasicFuture;
import org.apache.http.concurrent.FutureCallback;
import org.joda.time.DateTime;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.mail.yt.consumer.SourceConsumer;
import ru.yandex.search.mail.yt.consumer.YtClient;
import ru.yandex.search.mail.yt.consumer.YtException;
import ru.yandex.search.mail.yt.consumer.upload.AbstractYtTaskManager;
import ru.yandex.search.mail.yt.consumer.upload.JobContextBuilder;
import ru.yandex.search.mail.yt.consumer.upload.UploadWorkersManager;
import ru.yandex.tskv.TskvRecord;

public abstract class AbstractScheduler
    extends AbstractYtTaskManager
    implements Scheduler, Runnable, GenericAutoCloseable<IOException>
{
    private static final int SELECT_INTERVAL = 100;
    private static final int MAX_TASKS_PER_ITERATION = 5;
    private static final int MAX_SCHEDULED_JOBS = 30;

    protected final SourceConsumer sourceConsumer;
    protected final LockUpdater lockUpdater;
    private final Thread schedulerThread;
    private final Thread lockUpdaterThread;
    private final ConcurrentLinkedQueue<Runnable> jobCompleteQueue =
        new ConcurrentLinkedQueue<>();
    private final long checkIntervalMs;

    private long lastJobCheckTime = 0;
    private volatile boolean stop = false;

    protected AbstractScheduler(final SourceConsumer consumer)
        throws ConfigException
    {
        super(consumer, consumer.logger().addPrefix("Scheduler"));

        this.sourceConsumer = consumer;
        this.checkIntervalMs =
            TimeUnit.SECONDS.toMillis(
                this.sourceConsumer.config().checkInterval());

        this.lockUpdater = new LockUpdater(consumer);

        this.schedulerThread = new Thread(this);
        this.lockUpdaterThread = new Thread(lockUpdater);
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        Map<String, Map<String, String>> taskStat = new LinkedHashMap<>();
        List<SourceTask> completed;
        List<SourceTask> inProgress;
        synchronized (this) {
            completed = new ArrayList<>(this.completedSources);
            inProgress = new ArrayList<>(this.scheduledJobs.values());
        }
        for (SourceTask task: inProgress) {
            Map<String, String> stat = new LinkedHashMap<>();
            stat.put(COMPLETED, String.valueOf(task.completedCount()));
            stat.put("pending", String.valueOf(task.pendingCount()));
            taskStat.put(task.name(), stat);
        }

        status.put("running", taskStat);

        status.put(
            "completed",
            completed.stream().map(SourceTask::name)
                .collect(Collectors.toList()));

        status.put("MinimalSourceTimestamp", minTaskTs);

        return status;
    }

    @Override
    public void start() {
        this.lockUpdaterThread.start();
        this.schedulerThread.start();
    }

    protected abstract String extractName(final String name);

    protected SchedulerLock processNewSources(
        final SchedulerLock oldLock)
        throws InterruptedException
    {
        SchedulerLock lock = oldLock;
        logger.info("Waiting for lock");
        SchedulerLock newLock = this.lockUpdater.waitLock();
        long curTime = System.currentTimeMillis();
        logger.info("Got lock " + newLock.toString());
        try {
            if (newLock.locked() && newLock.master()) {
                if (!lock.locked() || !lock.master()) {
                    lock = newLock;
                    logger.info(
                        "Master or locked status achieved, "
                            + "loading status old "
                            + oldLock + " new " + newLock);

                    loadCurrentStatus();
                    return lock;
                }

                lock = newLock;

                if (curTime > lastJobCheckTime + checkIntervalMs) {
                    lastJobCheckTime = curTime;
                    if (scheduledJobs.size() >= MAX_SCHEDULED_JOBS) {
                        logger.warning(
                            "Current scheduled jobs count "
                                + scheduledJobs.size()
                                + " greater than maximum allowed "
                                + MAX_SCHEDULED_JOBS);
                    } else {
                        logger.info("Fetching new sources");
                        checkMonitorPath();
                        logger.info("New sources fetch finished");
                    }
                }

                if (trimSourcesRequested.compareAndSet(true, false)) {
                    trimSources();
                }
            } else {
                logger.info(
                    "Lock status " + lock.toString()
                        + " waiting to become master");
            }
        } catch (YtException | BadRequestException e) {
            logger.log(Level.WARNING, "Process new sources failed ", e);
        }

        return lock;
    }

    protected void processCompletedJobs() {
        while (!jobCompleteQueue.isEmpty()) {
            Runnable task = jobCompleteQueue.poll();
            if (task != null) {
                task.run();
            }
        }
    }

    @Override
    public void run() {
        SchedulerLock lock = MissingLock.INSTANCE;
        logger.info("Starting scheduler " + config.monitorPath());
        logger.info("Keeping completed " + config.keepCompletedCount());
        try {
            while (!stop) {
                processCompletedJobs();
                lock = processNewSources(lock);
                Thread.sleep(SELECT_INTERVAL);
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Scheduler interrupted", ie);
        }
    }

    @Override
    public List<String> jobs() {
        return new ArrayList<>(scheduledJobs.keySet());
    }

    @Override
    public Future<Job.JobStatus> completeJob(
        final String jobId,
        final PrefixedLogger suppliedLogger,
        final FutureCallback<Job.JobStatus> callback)
    {
        PrefixedLogger logger = suppliedLogger;
        if (logger == null) {
            logger = this.logger;
        }

        BasicFuture<Job.JobStatus> future = new BasicFuture<>(callback);
        jobCompleteQueue.add(new JobCompleteTask(future, logger, jobId));
        return future;
    }

    protected abstract DateTime accept(final String sourceName);

    /**
     * Checks monitor path for new files to upload
     * @throws BadRequestException
     * @throws YtException
     */
    protected void checkMonitorPath()
        throws BadRequestException, YtException, InterruptedException
    {
        List<SourceTask> candidates = new ArrayList<>();
        for (String name : yt.list(config.monitorPath())) {
            String sourceName = extractName(name);
            if (sourceName == null) {
                continue;
            }
            DateTime sourceDT = accept(sourceName);
            if (sourceDT == null) {
                logger.warning("Bad source name " + sourceName);
                continue;
            }

            String source = config.monitorPath() + '/' + name;
            long dateId = sourceDT.getMillis();
            String target =
                scheduledPath + '/' + sourceName + SCHEDULED_POSTFIX;

            boolean candidate;
            SourceTask task =
                new SourceTask(config, target, source, name, dateId);
            synchronized (this) {
                candidate = dateId > minTaskTs;
                candidate &= !scheduledSources.contains(task);
                candidate &= !completedSources.contains(task);
            }

            if (candidate) {
                candidates.add(task);
            }
        }

        Collections.sort(candidates);
        if (candidates.size() > MAX_TASKS_PER_ITERATION) {
            candidates = candidates.subList(0, MAX_TASKS_PER_ITERATION);
        }

        for (SourceTask task : candidates) {
            logger.info("Processing candidate " + task.name());
            int rows = yt.getLongAttribute(
                task.sourcePath(),
                YtClient.ROW_COUNT_ATTRIBUTE)
                .intValue();

            YtClient yt = this.yt.startTransaction();
            boolean success = false;
            try {
                yt.createTable(task.taskPath());

                int[][] ranges;
                if (rows > config.splitRowCount()) {
                    logger.info(
                        task.name() + " has size of "
                            + rows + " splitting");

                    int parts =
                        (int) Math.ceil(rows * 1.0 / config.splitRowCount());
                    int partSize = (int) Math.ceil(rows * 1.0 / parts);
                    ranges = new int[parts][];
                    int offset = 0;
                    for (int i = 0; i < parts; i++) {
                        int len = Math.min(partSize, rows - offset);
                        ranges[i] = new int[] {offset, len};
                        offset += len;
                    }
                } else {
                    ranges = new int[][] {new int[] {0, rows}};
                }

                logger.info("For source " + task.name()
                    + " created jobs count is " + ranges.length);

                for (int i = 0; i < ranges.length; i++) {
                    String jobId = String.valueOf(i) + '_' + task.name();
                    JobContextBuilder context =
                        new JobContextBuilder(yt, logger);
                    context.id(jobId);
                    context.consumer(config.consumer());
                    context.source(task.sourcePath());
                    context.offset(ranges[i][0]);
                    context.length(ranges[i][1]);
                    context.commitEvery(sourceConsumer.config().commitEvery());
                    context.scheduler(
                        sourceConsumer.currentHost().getHostName());

                    TskvRecord record = context.toTskv();

                    String jobPath =
                        sourceConsumer.workersConfig().todoPath() + '/' + jobId
                            + UploadWorkersManager.READY_FOR_PROCESS;

                    logger.info(
                        "Creating job file " + jobId + ' ' + jobPath
                            + " with offset, length " + context.offset()
                            + ' ' + context.length());

                    yt.createTable(jobPath);
                    yt.write(jobPath, record);
                    task.addJob(context);
                }

                logger.info("Creating schedule file " + task.taskPath());

                yt.write(task.taskPath(), task.toTskv());
                success = true;
            } finally {
                if (!success) {
                    yt.abortTransaction();
                } else {
                    yt.commitTransaction();
                }
            }

            synchronized (this) {
                for (String jobId: task.jobs().keySet()) {
                    scheduledJobs.put(jobId, task);
                }
                scheduledSources.add(task);
            }
        }
    }

    @Override
    public void close() throws IOException {
        stop = true;
    }

    @Override
    public SchedulerLock getMaster() {
        return lockUpdater.get();
    }

    private final class JobCompleteTask implements Runnable {
        private final BasicFuture<Job.JobStatus> future;
        private final PrefixedLogger logger;
        private final String jobId;

        private JobCompleteTask(
            final BasicFuture<Job.JobStatus> future,
            final PrefixedLogger logger,
            final String jobId)
        {
            this.future = future;
            this.logger = logger;
            this.jobId = jobId;
        }

        @Override
        @SuppressWarnings("FutureReturnValueIgnored")
        public void run() {
            try {
                SchedulerLock lock = lockUpdater.waitLock();

                logger.info(
                    "Completing job " + jobId
                        + " master is " + lock.lockedHost());

                if (!lock.master()) {
                    BasicAsyncRequestProducerGenerator generator =
                        new BasicAsyncRequestProducerGenerator(
                            "/jobCompleted?consumer="
                                + config.consumer().name()
                                + "&id=" + jobId);

                    while (!stop && !lock.master()) {
                        if (sourceConsumer.currentHost().equals(
                            lock.lockedHost()))
                        {
                            logger.info(
                                "Looks like we were master before reboot, "
                                    + "waiting real master");
                            Thread.sleep(checkIntervalMs);
                            lock = lockUpdater.waitLock();
                        } else {
                            break;
                        }
                    }

                    logger.info(
                        jobId + " redirecting job completion to "
                            + lock.lockedHost());
                    sourceConsumer.schedulersClient().execute(
                        lock.lockedHost(),
                        generator,
                        JobCompleteAsyncConsumerFactory.OK,
                        new JobCompleteCallback(future));
                } else {
                    future.completed(
                        AbstractScheduler.super.completeJob(logger, jobId));
                }
            } catch (InterruptedException ie) {
                future.failed(ie);
            }
        }
    }

    private static final class JobCompleteCallback
        implements FutureCallback<Job.JobStatus>
    {
        private final BasicFuture<Job.JobStatus> future;

        private JobCompleteCallback(final BasicFuture<Job.JobStatus> future) {
            this.future = future;
        }

        @Override
        public void completed(final Job.JobStatus jobStatus) {
            future.completed(jobStatus);
        }

        @Override
        public void failed(final Exception e) {
            future.failed(e);
        }

        @Override
        public void cancelled() {
            future.cancel();
        }
    }
}
