package ru.yandex.search.mail.yt.consumer.scheduler;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;

import org.apache.http.HttpHost;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.AsyncStringConsumerFactory;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mail.yt.consumer.SourceConsumer;
import ru.yandex.search.mail.yt.consumer.config.ImmutableSourceConsumerConfig;

public class LockUpdater
    implements Runnable, GenericAutoCloseable<IOException>
{
    private static final int INTERVAL_MULTIPLIER = 6;
    private static final String ID_PARAM = "&id=";

    private final ImmutableSourceConsumerConfig config;
    private final PrefixedLogger logger;
    private final AsyncClient producerClient;
    private final HttpHost currentHost;
    private final Set<HttpHost> schedulers;
    private final AsyncClient schedulersClient;

    private final AtomicBoolean updating = new AtomicBoolean(false);
    private final AtomicReference<SchedulerLock> lock
        = new AtomicReference<>(MissingLock.INSTANCE);

    private volatile boolean stop = false;

    public LockUpdater(final SourceConsumer sourceConsumer) {
        this.config = sourceConsumer.config();
        this.schedulers = config.schedulers();
        this.currentHost = sourceConsumer.currentHost();
        this.logger =
            sourceConsumer.logger().addPrefix("LockUpdater");

        this.producerClient = sourceConsumer.producer();
        this.schedulersClient = sourceConsumer.schedulersClient();
    }

    @Override
    public void close() throws IOException {
        stop = true;
    }

    public SchedulerLock get() {
        return this.lock.get();
    }

    public SchedulerLock waitLock() throws InterruptedException {
        while (!stop) {
            SchedulerLock lock = this.lock.get();
            if (lock.locked()) {
                return lock;
            } else {
                updateLockStatus();
            }
        }

        return lock.get();
    }

    @Override
    public void run() {
        try {
            while (!stop) {
                SchedulerLock lock = this.lock.get();
                logger.info(
                    "Check lock status " + lock.toString()
                        + ' ' + updating.get());
                if (!lock.locked() || lock.leftAlive() < lock.timeout() / 2) {
                    updateLockStatus();
                    logger.info(
                        "Lock status updated " + this.lock.get().toString());
                } else {
                    logger.info("Time not come yet " + lock.leftAlive());
                }

                Thread.sleep(config.lockTimeout() / INTERVAL_MULTIPLIER);
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Update lock interrupted", ie);
        }
    }

    private void updateLockStatus() throws InterruptedException {
        while (true) {
            if (updating.compareAndSet(false, true)) {
                SchedulerLock lock = this.lock.get();
                logger.info("Starting lock update " + lock);
                if (!lock.locked()) {
                    lock = acquireOtherStatus();
                    logger.info(lock.toString() + " from other nodes");
                }

                // if we are master and
                if (lock.master() || !lock.locked()) {
                    lock = producerLock(lock);
                }

                this.lock.set(lock);
                synchronized (this) {
                    updating.set(false);
                    this.notifyAll();
                }
            } else {
                logger.info("Somebody already updating lock, waiting");
                synchronized (this) {
                    if (updating.get()) {
                        this.wait();
                    }
                }
            }

            if (lock.get().locked()) {
                break;
            }

            Thread.sleep(config.lockTimeout() / INTERVAL_MULTIPLIER);
        }
    }

    private SchedulerLock acquireOtherStatus() {
        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(
                "/isMaster?consumer=" + config.consumer().name());

        Map<HttpHost, Future<JsonObject>> futures =
            new LinkedHashMap<>(schedulers.size());
        for (HttpHost scheduler: schedulers) {
            if (currentHost.equals(scheduler)) {
                continue;
            }

            futures.put(
                scheduler,
                schedulersClient.execute(
                    scheduler,
                    generator,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    EmptyFutureCallback.INSTANCE));
        }

        int aliveCount = 0;
        logger.info(futures.size() + " requests to nodes, sent");
        SchedulerLock lock = MissingLock.INSTANCE;
        for (Map.Entry<HttpHost, Future<JsonObject>> entry
            : futures.entrySet())
        {
            try {
                lock = SchedulerLock.parse(entry.getValue().get());
                aliveCount += 1;

                if (lock.locked()) {
                    break;
                }
            } catch (ExecutionException e) {
                logger.log(
                    Level.WARNING,
                    "Scheduler request failed",
                    e);
            } catch (JsonException je) {
                logger.log(
                    Level.WARNING,
                    "Bad lock format",
                    je);
            } catch (InterruptedException ie) {
                logger.log(
                    Level.WARNING,
                    "Scedulers request interrupted",
                    ie);
            }
        }

        if (lock.locked()) {
            logger.info("Have locked scheduler " + lock.toString());
        } else {
            logger.info(
                "No other schedulers are locked, alive "
                    + aliveCount + ' ' + lock.toString());
        }

        return lock;
    }

    private SchedulerLock producerLock(final SchedulerLock lock) {
        String lockUri =
            "/_lock?service=" + config.service()
                + "&name=scheduler-" + config.consumer().name()
                + "&timeout=" + config.lockTimeout();

        boolean alreadyLocked = lock.locked() && lock.master();

        if (alreadyLocked) {
            lockUri += ID_PARAM + lock.lockId();
        }

        logger.info(
            "Trying to get/update producer lock, current "
                + lock + " uri " + lockUri);

        Future<String> future = producerClient.execute(
            config.producer().host(),
            new BasicAsyncRequestProducerGenerator(lockUri),
            AsyncStringConsumerFactory.OK,
            EmptyFutureCallback.INSTANCE);

        SchedulerLock newLock = lock;
        try {
            newLock = new MasterLock(config, future.get(), currentHost);
            logger.info("Got lock status " + newLock.toString());
        } catch (ExecutionException ee) {
            logger.info("Failed to obtain lock " + ee.getMessage());
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Obtain lock interrupted", ie);
        }

        if (!newLock.locked()) {
            newLock = MissingLock.INSTANCE;
        }

        return newLock;
    }
}
