package ru.yandex.search.mail.yt.consumer.scheduler;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mail.yt.consumer.YtClient;
import ru.yandex.search.mail.yt.consumer.config.SourceConsumerConfig;
import ru.yandex.search.mail.yt.consumer.upload.JobContextBuilder;
import ru.yandex.tskv.TskvException;
import ru.yandex.tskv.TskvRecord;

public class SourceTask implements Comparable<SourceTask> {
    //job fields
    protected static final String JOBID = "id";
    protected static final String STATUS = "status";
    protected static final String SOURCE = "source";
    protected static final String OFFSET = "offset";
    protected static final String LENGTH = "length";
    protected static final String START_TIME = "start_time";
    protected static final String FINISH_TIME = "finish_time";
    //task  fields
    private static final String NAME = "name";
    private static final String DATE_ID = "dateId";

    private final String sourcePath;
    private final String name;
    private final String taskPath;
    private final String completedPath;
    private final long dateId;

    private final Map<String, Job> jobs = new LinkedHashMap<>();

    // CSOFF: ParameterNumber
    public SourceTask(
        final SourceConsumerConfig config,
        final String taskPath,
        final String sourcePath,
        final String name,
        final long dateId)
    {
        this.taskPath = taskPath;
        this.name = name;
        this.dateId = dateId;
        this.sourcePath = sourcePath;
        this.completedPath =
            config.basePath() + '/' + AbstractScheduler.COMPLETED
                + '/' + name + AbstractScheduler.COMPLETED_POSTFIX;
    }
    // CSON: ParameterNumber

    public long completedCount() {
        return jobs.values().stream().filter(
            (v) -> v.status() == Job.JobStatus.COMPLETED).count();
    }

    public long pendingCount() {
        return jobs.values().stream().filter(
            (v) -> v.status() != Job.JobStatus.COMPLETED).count();
    }

    public String sourcePath() {
        return sourcePath;
    }

    public String name() {
        return name;
    }

    public String taskPath() {
        return taskPath;
    }

    public Map<String, Job> jobs() {
        return jobs;
    }

    public String completedPath() {
        return completedPath;
    }

    public synchronized boolean completed() {
        return jobs.values().stream().allMatch(
            v -> v.status() == Job.JobStatus.COMPLETED);
    }

    @Override
    public String toString() {
        return "SourceTask{"
            + "sourcePath='" + sourcePath + '\''
            + ", name='" + name + '\''
            + ", taskPath='" + taskPath + '\''
            + ", dateId=" + dateId
            + '}';
    }

    // CSOFF: ReturnCount
    public Job.JobStatus completeJob(
        final String jobId,
        final YtClient yt,
        final PrefixedLogger logger)
        throws InterruptedException
    {
        Job job = jobs.get(jobId);
        if (job == null) {
            logger.warning(jobId + " not found for " + this.toString());
            return Job.JobStatus.NOT_EXISTS;
        }

        if (!job.statusRef().compareAndSet(
            Job.JobStatus.SCHEDULED,
            Job.JobStatus.COMPLETING))
        {
            logger.info(
                jobId + " skipping complete, state is " + job.status());
            return job.status();
        }

        yt.schedule((y) -> {
            logger.info(
                "Saving progress for " + taskPath() + ' ' + jobId);
            if (!yt.exists(taskPath())) {
                logger.severe("Missing file " + taskPath());
            } else {
                yt.write(taskPath(), toTskv());
            }
            return null;
        });

        logger.info(jobId + " status saved");
        job.statusRef().set(Job.JobStatus.COMPLETED);
        return job.status();
    }
    // CSON: ReturnCount

    public synchronized boolean hasJob(final String jobId) {
        return jobs.containsKey(jobId);
    }

    public long dateId() {
        return dateId;
    }

    public Job addJob(final JobContextBuilder context) {
        Job job =
            new Job(
                context.id(),
                context.source(),
                Job.JobStatus.SCHEDULED,
                context.offset(),
                context.length(),
                System.currentTimeMillis(),
                -1L);
        jobs.put(job.id(), job);
        return job;
    }

    public List<TskvRecord> toTskv() {
        List<TskvRecord> result = new ArrayList<>(jobs.size() + 1);
        TskvRecord meta = new TskvRecord();
        meta.put(NAME, name());
        meta.put(DATE_ID, dateId());
        meta.put(SOURCE, sourcePath());
        result.add(meta);
        result.addAll(jobs.values().stream().map(Job::toTskv)
            .collect(Collectors.toList()));
        return result;
    }

    public static SourceTask fromFile(
        final SourceConsumerConfig config,
        final YtClient yt,
        final String path)
        throws TskvException, InterruptedException
    {
        List<TskvRecord> records = yt.schedule((c) -> c.readTskv(path));
        if (records == null) {
            return null;
        }

        if (records.size() < 1) {
            throw new TskvException("Invalid source task size");
        }
        //first record is meta about task
        TskvRecord taskRecord = records.get(0);

        String name = taskRecord.getString(NAME);
        long dateId = taskRecord.getLong(DATE_ID);
        String source = taskRecord.getString(SOURCE);

        SourceTask task =
            new SourceTask(
                config,
                path,
                source,
                name,
                dateId);

        for (int i = 1; i < records.size(); ++i) {
            Job job = jobFromRecord(records.get(i));
            task.jobs.put(job.id(), job);
        }

        return task;
    }

    @Override
    public int compareTo(final SourceTask o) {
        return Long.compare(this.dateId, o.dateId);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof SourceTask)) {
            return false;
        }

        SourceTask task = (SourceTask) o;

        if (dateId != task.dateId) {
            return false;
        }
        return sourcePath.equals(task.sourcePath);
    }

    @Override
    public int hashCode() {
        return sourcePath.hashCode();
    }

    private static Job jobFromRecord(
        final TskvRecord record)
        throws TskvException
    {
        String id = record.get(JOBID);
        String source = record.get(SOURCE);

        Job.JobStatus status;
        try {
            status = record.getEnum(Job.JobStatus.class, STATUS);
        } catch (TskvException e) {
            boolean bool = record.getBoolean(STATUS, false);
            if (bool) {
                status = Job.JobStatus.COMPLETED;
            } else {
                status = Job.JobStatus.SCHEDULED;
            }
        }

        if (status == Job.JobStatus.COMPLETING) {
            status = Job.JobStatus.COMPLETED;
        }

        long offset = record.getLong(OFFSET);
        long length = record.getLong(LENGTH);
        long startTime = record.getLong(START_TIME);
        long finishTime = record.getLong(FINISH_TIME, -1L);
        return new Job(
            id,
            source,
            status,
            offset,
            length,
            startTime,
            finishTime);
    }
}
