package ru.yandex.search.mail.yt.consumer.upload;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.logging.Level;

import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;

import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.search.mail.yt.consumer.QueueData;
import ru.yandex.search.mail.yt.consumer.SourceConsumer;
import ru.yandex.search.mail.yt.consumer.YtException;

public abstract class AbstractUploader<T>
    implements Uploader, Function<T, Boolean>
{
    private static final int DEFAULT_SAVE_MLT = 100;

    protected final BasicJobContext context;
    protected final AsyncClient producer;
    protected final HttpHost producerHost;
    protected final QueueAcceptor acceptor;
    protected final AtomicInteger readRecords = new AtomicInteger(0);

    protected volatile boolean stop = false;

    protected AbstractUploader(
        final SourceConsumer sourceConsumer,
        final BasicJobContext context)
    {
        this.context = context;
        this.producer = sourceConsumer.producer();
        this.producerHost = sourceConsumer.config().producer().host();

        this.acceptor = new QueueAcceptor(batchSize());
    }

    @Override
    public Boolean apply(final T record) {
        if (stop) {
            return true;
        }

        parse(record, acceptor);
        readRecords.incrementAndGet();

        boolean status = context.length() <= 0;
        if (!stop) {
            if (acceptor.readyToFlush()
                || acceptor.consumed()
                >= DEFAULT_SAVE_MLT * acceptor.batchSize)
            {
                status = true;
            }
        }

        return status;
    }

    protected abstract ReadingYtClient<T> yt();

    @Override
    public void upload() throws InterruptedException {
        while (!stop && context.length() > 0) {
            int offset = context.offset();
            int len =
                Math.min(
                    context.length(),
                    DEFAULT_SAVE_MLT * acceptor.batchSize);

            context.logger().info(
                "Reading offset " + offset + " length " + len);
            readRecords.set(0);
            try {
                yt().read(context.source(), offset, len);
                push();
            } catch (YtException ye) {
                context.logger().log(
                    Level.WARNING,
                    "Failed to upload " + context.toString(),
                    ye);
                push(true);
            }

            context.logger().info(
                "Records read " + readRecords.get()
                    + " offset" + context.offset());

            Thread.sleep(DEFAULT_SAVE_MLT);
        }

        push(true);
    }

    @Override
    public void close() {
        stop = true;
    }

    protected void push() {
        push(false);
    }

    //CSOFF: ReturnCount
    protected void push(final boolean forceCommit) {
        context.logger().info(
            "Pushing to queue, size is " + acceptor.size()
                + " consumed " + acceptor.consumed()
                + " batch_size " + acceptor.batchSize);

        if (acceptor.size() <= 0) {
            context.logger().warning("Skipping empty queue");
            context.saveProgress(acceptor.consumed(), forceCommit);
            return;
        }

        List<QueueData> queue = acceptor.data();

        QueueData first = queue.get(0);
        HttpPost request;
        if (queue.size() == 1) {
            request = new HttpPost(first.uri());
            request.setEntity(
                new StringEntity(
                    first.body(),
                    ContentType.APPLICATION_JSON));

            request.addHeader(
                YandexHeaders.ZOO_SHARD_ID,
                Integer.toString(first.shard()));
        } else {
            StringBuilder uri = new StringBuilder("/notify?");
            uri.append("service");
            uri.append('=');
            uri.append(first.service());
            uri.append("&first-operation-date=");
            uri.append((long) first.operationDate());
            uri.append("&batch-size=");
            uri.append(queue.size());
            uri.append("&consumer=ytConsumer");
            uri.append("&path=");
            uri.append(context.source());
            uri.append("&jobId=");
            uri.append(context.id());
            uri.append("&ytPosition=");
            uri.append(context.offset());

            MultipartEntityBuilder builder = MultipartEntityBuilder.create();
            builder.setMimeSubtype("mixed");
            final String envelopeName = "envelope.json";

            for (QueueData queueData: queue) {
                builder.addPart(
                    FormBodyPartBuilder
                        .create()
                        .addField(
                            YandexHeaders.ZOO_SHARD_ID,
                            Integer.toString(queueData.shard()))
                        .addField(YandexHeaders.URI, queueData.uri())
                        .setBody(
                            new ByteArrayBody(
                                queueData.body().getBytes(
                                    Charset.forName("utf-8")),
                                ContentType.APPLICATION_JSON,
                                envelopeName))
                        .setName(envelopeName)
                        .build());
            }
            request = new HttpPost(uri.toString());
            request.setEntity(builder.build());
        }

        request.addHeader(YandexHeaders.SERVICE, first.service());

        try {
            send(request);
            context.saveProgress(acceptor.consumed(), forceCommit);
            acceptor.clear();
        } catch (InterruptedException ie) {
            context.logger().log(
                Level.WARNING,
                "Sending batch interrupted " + request.toString(),
                ie);
        }
    }
    //CSON: ReturnCount

    protected void send(final HttpRequest request) throws InterruptedException {
        while (!stop) {
            try {
                Future<HttpResponse> future =
                    producer.execute(
                        producerHost,
                        new BasicAsyncRequestProducerGenerator(request),
                        EmptyFutureCallback.INSTANCE);
                HttpResponse response = future.get();
                int statusCode = response.getStatusLine().getStatusCode();
                if (statusCode != HttpStatus.SC_OK
                    && statusCode != HttpStatus.SC_ACCEPTED)
                {
                    context.logger().info(
                        "Producer request " + request.toString()
                            + " failed with code " + statusCode);
                } else {
                    context.logger().info(
                        "Producer data sent " + request.toString());
                    return;
                }
            } catch (ExecutionException | IOException ioe) {
                context.logger().log(
                    Level.WARNING,
                    "Producer request failed " + request.toString(),
                    ioe);
            }

            Thread.sleep(failureSendDelay());
            context.logger().info(
                "Another attempt to deliver batch to queue");
        }
    }

    protected abstract void parse(
        final T record,
        final QueueAcceptor acceptor);

    protected abstract int batchSize();

    protected abstract int failureSendDelay();
}
