package ru.yandex.search.mail.yt.consumer.upload;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;

import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.search.mail.yt.consumer.SourceConsumer;
import ru.yandex.search.mail.yt.consumer.YtClient;
import ru.yandex.search.mail.yt.consumer.YtException;
import ru.yandex.search.mail.yt.consumer.config.ImmutableSourceConsumerConfig;
import ru.yandex.search.mail.yt.consumer.scheduler.Job;
import ru.yandex.search.mail.yt.consumer.scheduler.SourceTask;
import ru.yandex.tskv.TskvException;

public abstract class AbstractYtTaskManager {
    public static final String SCHEDULED = "scheduled";
    public static final String SCHEDULED_POSTFIX = '.' + SCHEDULED;
    public static final String COMPLETED = "completed";
    public static final String COMPLETED_POSTFIX = '.' + COMPLETED;

    // should be guarded by this
    protected final List<SourceTask> completedSources = new ArrayList<>();
    protected final Set<SourceTask> scheduledSources = new HashSet<>();
    //
    protected final ConcurrentHashMap<String, SourceTask> scheduledJobs =
        new ConcurrentHashMap<>();
    protected final AtomicBoolean trimSourcesRequested =
        new AtomicBoolean(false);
    protected final ImmutableSourceConsumerConfig config;
    protected final PrefixedLogger logger;
    protected final YtClient yt;
    protected final String completedPath;
    protected final String scheduledPath;

    protected volatile long minTaskTs;

    protected AbstractYtTaskManager(
        final SourceConsumer sourceConsumer,
        final PrefixedLogger logger)
        throws ConfigException
    {
        this.config = sourceConsumer.config();
        this.logger = logger;
        this.yt = sourceConsumer.yt();

        this.completedPath = config.basePath() + '/' + COMPLETED;
        this.scheduledPath = config.basePath() + '/' + SCHEDULED;
        this.minTaskTs = minimalTs();
    }

    protected synchronized void loadCurrentStatus()
        throws YtException, InterruptedException
    {
        logger.info("Loading tasks status from disk");
        scheduledSources.clear();
        scheduledJobs.clear();
        completedSources.clear();

        if (!yt.exists(this.completedPath)) {
            yt.createDirectory(completedPath, true);
        }

        if (!yt.exists(this.scheduledPath)) {
            yt.createDirectory(scheduledPath, true);
        }

        if (!yt.exists(this.config.monitorPath())) {
            throw new YtException(
                "Monitor path do not exists " + this.config.monitorPath());
        }

        for (String name: yt.list(this.scheduledPath)) {
            String path = this.scheduledPath + '/' + name;

            SourceTask task;
            try {
                task = SourceTask.fromFile(config, yt, path);
            } catch (TskvException e) {
                logger.log(Level.WARNING, "Failed to load " + path, e);
                logger.log(
                    Level.WARNING,
                    "Cleaning task that failed to load " + path,
                    e);
                yt.schedule((yt) -> {
                    if (yt.exists(path)) {
                        yt.remove(path);
                    }

                    return null;
                });
                continue;
            }

            if (name.endsWith(SCHEDULED_POSTFIX) && task != null) {
                for (Job job: task.jobs().values()) {
                    if (job.status() != Job.JobStatus.COMPLETED) {
                        scheduledJobs.put(job.id(), task);
                    }
                }
                scheduledSources.add(task);
            } else if (name.endsWith(COMPLETED_POSTFIX) && task != null) {
                completedSources.add(task);
            }
        }

        for (String name: yt.list(this.completedPath)) {
            String path = this.completedPath + '/' + name;

            if (name.endsWith(COMPLETED_POSTFIX)) {
                SourceTask task;
                try {
                    task = SourceTask.fromFile(config, yt, path);
                } catch (TskvException e) {
                    logger.log(
                        Level.WARNING,
                        "Failed to load completed " + path,
                        e);
                    continue;
                }

                completedSources.add(task);
            }
        }

        //check if some tasks completed but not moved
        for (SourceTask task: new ArrayList<>(scheduledSources)) {
            if (task.completed()) {
                logger.info(
                    "Task completed but not moved " + task.toString());
                completedTask(task);
            }
        }

        updateMinimalTs();
        trimSources();
    }

    protected Job.JobStatus completeJob(
        final PrefixedLogger logger,
        final String jobId)
        throws InterruptedException
    {
        Job.JobStatus result;

        SourceTask jobTask = scheduledJobs.get(jobId);
        if (jobTask == null) {
            result = Job.JobStatus.NOT_EXISTS;
            logger.warning("No job found " + jobId);
            logger.info("Current jobs " + scheduledJobs.keySet());
        } else {
            result = jobTask.completeJob(jobId, yt, logger);
            if (result == Job.JobStatus.COMPLETED) {
                scheduledJobs.remove(jobId);

                if (jobTask.completed()) {
                    completedTask(jobTask);
                }
            }
        }

        return result;
    }

    protected long minimalTs() {
        return 0L;
    }

    protected synchronized void updateMinimalTs() {
        Optional<Long> processingMinTs =
            scheduledJobs.values().stream()
                .map(SourceTask::dateId).min(Long::compareTo);
        Optional<Long> completedMinTs =
            completedSources.stream()
                .map(SourceTask::dateId).min(Long::compareTo);
        long minTs = Long.MAX_VALUE;
        if (processingMinTs.isPresent()) {
            minTs = processingMinTs.get();
        }

        if (completedMinTs.isPresent() && completedMinTs.get() < minTs) {
            minTs = completedMinTs.get();
        }

        if (minTs < minimalTs() || minTs == Long.MAX_VALUE) {
            minTs = minimalTs();
        }

        logger.info(
            "New minimal ts was "
                + minTaskTs + " change to " + minTs);
        this.minTaskTs = minTs;
    }

    protected void trimSources() throws InterruptedException {
        List<SourceTask> sourcesToDelete = new ArrayList<>();

        synchronized (this) {
            if (this.completedSources.size() <= config.keepCompletedCount()) {
                return;
            }

            Optional<Long> processingMinTs =
                scheduledJobs.values().stream()
                    .map(SourceTask::dateId).min(Long::compareTo);

            completedSources.sort(Comparator.reverseOrder());

            List<SourceTask> sources = new ArrayList<>();

            for (int i = 0; i < completedSources.size(); i++) {
                if (i < config.keepCompletedCount()) {
                    sources.add(completedSources.get(i));
                } else {
                    SourceTask taskForDelete = completedSources.get(i);
                    if (!processingMinTs.isPresent()
                        || taskForDelete.dateId() <= processingMinTs.get())
                    {
                        sourcesToDelete.add(taskForDelete);
                    } else {
                        sources.add(taskForDelete);
                    }
                }
            }

            this.completedSources.clear();
            this.completedSources.addAll(sources);
            updateMinimalTs();
        }

        logger.info("After trimming " + completedSources);

        for (SourceTask task: sourcesToDelete) {
            logger.info(
                "Removing stale completed task " + task.completedPath());

            yt.schedule((y) -> {
                if (yt.exists(task.completedPath())) {
                    yt.remove(task.completedPath());
                    logger.info("Removed " + task.completedPath());
                } else {
                    logger.warning(
                        "Task to remove not found " + task.completedPath());
                }

                return null;
            });
        }
    }

    protected boolean completedTask(
        final SourceTask task)
        throws InterruptedException
    {
        boolean done = false;

        synchronized (this) {
            scheduledSources.remove(task);
            if (!completedSources.contains(task)) {
                completedSources.add(task);
                done = true;
            }
        }

        logger.info("After completing task " + completedSources);

        if (!done) {
            return false;
        }

        logger.info("Removing finished task " + task.sourcePath());
        yt.schedule((yt) -> {
            logger.fine(
                "Moving " + task.taskPath()
                    + " to " + task.completedPath());

            boolean sourceExists = yt.exists(task.taskPath());
            boolean targetExists = yt.exists(task.completedPath());
            if (sourceExists && !targetExists) {
                yt.move(task.taskPath(), task.completedPath());
                logger.info(
                    "Task file " + task.taskPath()
                        + " moved to " + task.completedPath());
            } else {
                if (!sourceExists) {
                    logger.warning(
                        "Task not exists, skip move " + task.taskPath());
                }

                if (targetExists) {
                    logger.warning(
                        task.completedPath() + " already exists removing "
                            + task.taskPath());
                    yt.remove(task.taskPath());
                }
            }

            return null;
        });

        trimSourcesRequested.compareAndSet(false, true);
        return true;
    }
}
