package ru.yandex.search.mail.yt.consumer.upload;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mail.yt.consumer.YtClient;
import ru.yandex.search.mail.yt.consumer.YtException;
import ru.yandex.tskv.TskvRecord;

public class BasicJobContext
    implements JobContext, GenericAutoCloseable<IOException>
{
    private static final int UPDATE_OFFSET_INTERVAL = 300;

    private final String id;
    private final String workerFile;
    private final String completedPath;
    private final String malformedPath;
    private final String source;
    private final String scheduler;
    private final int commitEvery;
    private final AtomicBoolean saving = new AtomicBoolean(false);
    private final SourceConsumerFactory consumer;

    private final YtClient client;

    private volatile boolean stop;

    private PrefixedLogger logger;

    private final AtomicInteger offset;
    private final AtomicInteger length;
    private final AtomicInteger sinceLastCommit;

    public BasicJobContext(
        final JobContext context,
        final YtClient client,
        final PrefixedLogger logger)
    {
        this.client = client;
        this.id = context.id();
        this.workerFile = context.workerFile();
        this.completedPath = context.completedPath();
        this.malformedPath = context.malformedPath();
        this.source = context.source();
        this.scheduler = context.scheduler();
        this.consumer = context.consumer();
        this.commitEvery = context.commitEvery();

        this.offset = new AtomicInteger(context.offset());
        this.length = new AtomicInteger(context.length());
        this.sinceLastCommit = new AtomicInteger(0);

        this.logger = logger.addPrefix(id);
    }

    public void saveProgress(final int consumed, final boolean force) {
        int value = sinceLastCommit.addAndGet(consumed);
        if ((force && value <= 0) || (!force && value < commitEvery())) {
            offset.addAndGet(consumed);
            length.addAndGet(-consumed);
            return;
        }

        try {
            while (!stop) {
                if (saving.compareAndSet(false, true)) {
                    break;
                }

                Thread.sleep(UPDATE_OFFSET_INTERVAL);
            }
        } catch (InterruptedException ie) {
            logger.log(
                Level.WARNING,
                "Failed to save progress " + consumed,
                ie);
            return;
        }

        int newOffset = offset.get() + consumed;
        int newLength = length.get() - consumed;

        TskvRecord record = toTskv();
        record.put(OFFSET, newOffset);
        record.put(LENGTH, newLength);

        boolean success = false;
        try {
            while (!stop) {
                try {
                    client.write(workerFile, record);
                    success = true;
                    break;
                } catch (YtException e) {
                    logger.log(
                        Level.WARNING,
                        "Failed to update progress " + consumed,
                        e);
                }

                Thread.sleep(UPDATE_OFFSET_INTERVAL);
                logger.info("Another attempt to save offset");
            }

            if (success) {
                sinceLastCommit.set(0);
                logger.info(
                    "Position updated from " + offset + ' ' + length
                        + " to " + newOffset + ' ' + newLength);
                offset.set(newOffset);
                length.set(newLength);
            } else {
                logger.warning("Progress update failed");
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Saving offset interrupted", ie);
        } finally {
            saving.set(false);
        }
    }

    @Override
    public SourceConsumerFactory consumer() {
        return consumer;
    }

    @Override
    public String id() {
        return id;
    }

    @Override
    public String workerFile() {
        return workerFile;
    }

    @Override
    public String completedPath() {
        return completedPath;
    }

    @Override
    public String malformedPath() {
        return malformedPath;
    }

    @Override
    public String source() {
        return source;
    }

    @Override
    public String scheduler() {
        return scheduler;
    }

    @Override
    public PrefixedLogger logger() {
        return logger;
    }

    @Override
    public int offset() {
        return offset.get();
    }

    @Override
    public int length() {
        return length.get();
    }

    @Override
    public void close() throws IOException {
        stop = true;
    }

    @Override
    public int commitEvery() {
        return commitEvery;
    }
}
