package ru.yandex.search.mail.yt.consumer.upload;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.function.Function;

import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.search.mail.yt.consumer.JsonStreamAsyncConsumerFactory;
import ru.yandex.search.mail.yt.consumer.YtClient;
import ru.yandex.search.mail.yt.consumer.YtException;

public class JsonReadingYtClient
    extends ReadingYtClient<JsonObject>
{
    private static final int BAD_RECORD_PRECISION = 100;
    private static final JsonMap MALFORMED_RECORD =
        new JsonMap(BasicContainerFactory.INSTANCE);
    private final StatusCheckAsyncResponseConsumerFactory<Void> consumerFactory;
    private final Function<JsonObject, Boolean> consumer;

    public JsonReadingYtClient(
        final YtClient yt,
        final Function<JsonObject, Boolean> consumer)
    {
        super(yt);

        this.consumerFactory =
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                new JsonStreamAsyncConsumerFactory(consumer));
        this.consumer = consumer;
    }

    protected Future<Void> readInternal(
        final String path,
        final int offset,
        final int length)
        throws YtException, InterruptedException
    {
        List<Header> headers = new ArrayList<>();
        headers.add(
            new BasicHeader(HttpHeaders.ACCEPT, "application/json"));

        if (length >= 0) {
            headers.add(
                new BasicHeader(
                    "X-YT-Output-Format",
                    "{\"$value\":\"json\","
                        + "\"$attributes\":{\"encode_utf8\":false}}"));
            headers.add(
                new BasicHeader(
                    YtClient.YT_PARAMS_HEADER,
                    "{\"path\":{\"$value\":\"" + path
                        + "\",\"$attributes\":{\"ranges\":["
                        + "{\"lower_limit\":{\"row_index\":" + offset
                        + "}, \"upper_limit\":{\"row_index\":"
                        + (offset + length) + "}}]}}}"));
        }

        yt.logger().info(
            "Reading " + path + '[' + offset
                + ',' + (offset + length));
        return yt.read(
            path,
            headers,
            consumerFactory,
            EmptyFutureCallback.INSTANCE);
    }

    @Override
    public void read(
        final String path,
        final int offset,
        final int length)
        throws InterruptedException, YtException
    {
        int step = length;
        int stepOffset = offset;
        int threshold = offset + length;
        while (stepOffset < threshold) {
            if (stepOffset + step >= threshold) {
                step = threshold - stepOffset;
            }

            Future<Void> future = readInternal(path, stepOffset, step);
            try {
                future.get();
                stepOffset += step;
                if (step <= BAD_RECORD_PRECISION) {
                    step = threshold - stepOffset;
                }
            } catch (ExecutionException ee) {
                if (checkMalformedJson(ee.getCause())) {
                    if (step > BAD_RECORD_PRECISION) {
                        step = BAD_RECORD_PRECISION;
                        yt.logger().info(
                            "Bad records in " + path + '[' + stepOffset
                                + ',' + (stepOffset + step));
                    } else {
                        yt.logger().warning(
                            "Skipping interval " + stepOffset
                                + ',' + (stepOffset + step));

                        for (int i = 0; i < step; i++) {
                            if (consumer.apply(MALFORMED_RECORD)) {
                                break;
                            }
                        }

                        stepOffset += step;
                    }
                } else {
                    throw new YtException(ee.getCause());
                }
            }
        }
    }

    private boolean checkMalformedJson(final Throwable t) {
        if (t instanceof BadResponseException) {
            return ((BadResponseException) t).response()
                .contains("Invalid floating point value in json");
        }

        return false;
    }
}
