package ru.yandex.search.mail.yt.consumer.upload;

import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;

import org.apache.http.message.BasicHttpRequest;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mail.yt.consumer.YtClient;
import ru.yandex.search.mail.yt.consumer.YtConsumer;
import ru.yandex.search.mail.yt.consumer.YtException;
import ru.yandex.search.mail.yt.consumer.scheduler.Job;
import ru.yandex.tskv.TskvException;
import ru.yandex.tskv.TskvRecord;

public class UploadWorker
    implements Runnable, GenericAutoCloseable<IOException>
{
    public static final String WORKERS_LOGGER = "/workers";
    private final PrefixedLogger logger;
    private final YtClient yt;
    private final YtConsumer ytConsumer;
    private final UploadTask task;

    private volatile Uploader uploader = null;

    public UploadWorker(
        final UploadTask task,
        final YtClient ytClient,
        final YtConsumer ytConsumer)
    {
        this.logger =
            ytConsumer.config().loggers().preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        UploadWorker.WORKERS_LOGGER)));
        this.yt = ytClient.adjustLogger(logger.addPrefix("YT-CLIENT"));
        this.task = task;

        this.ytConsumer = ytConsumer;
    }

    @Override
    public void close() throws IOException {
        if (uploader != null) {
            uploader.close();
        }
    }

    // CSOFF: ReturnCount
    @Override
    public void run() {
        try {
            logger.info("Starting job for " + task.processingPath());
            if (!yt.exists(task.processingPath())) {
                logger.severe("Path do not exists " + task.processingPath());
                return;
            }

            TskvRecord record = yt.readTskvFirst(task.processingPath());
            if (record == null) {
                logger.warning("No jobs in " + task.processingPath());
                yt.move(task.processingPath(), task.malformedPath());
                return;
            }

            JobContextBuilder contextBuilder;
            try {
                contextBuilder =
                    JobContextBuilder.create(yt, logger, record);
                contextBuilder.workerFile(task.processingPath());

                logger.info("Context " + contextBuilder.toTskv());
            } catch (TskvException e) {
                logger.log(
                    Level.WARNING,
                    "Malformed job " + record.toString(),
                    e);
                yt.move(task.processingPath(), task.malformedPath());
                return;
            }

            int rows = yt.getLongAttribute(
                contextBuilder.source(),
                YtClient.ROW_COUNT_ATTRIBUTE)
                .intValue();
            if (rows < contextBuilder.offset() + contextBuilder.length()) {
                int trimmedLength =
                    Math.max(0, rows - contextBuilder.offset());
                logger.warning(
                    "Size of file is less than offset + length, size: "
                        + rows + " offset " + contextBuilder.offset()
                        + " length " + contextBuilder.length()
                        + " trimming to " + trimmedLength);
                contextBuilder.length(trimmedLength);
            }

            BasicJobContext context = contextBuilder.build();
            uploader =
                ytConsumer.uploader(context.consumer()).create(context);

            logger.info(
                "Context for worker is "
                    + context.toTskv().toString());

            uploader.upload();

            logger.info("Completing " + context.toTskv().toString());
            try {
                while (true) {
                    try {
                        Future<Job.JobStatus> future =
                            ytConsumer.scheduler(context.consumer())
                                .completeJob(context.id(), logger, null);
                        Job.JobStatus status = future.get();
                        if (status == Job.JobStatus.COMPLETED
                            || status == Job.JobStatus.NOT_EXISTS)
                        {
                            logger.info(
                                context.id()
                                    + " completed with status " + status);
                            break;
                        }
                    } catch (ExecutionException ee) {
                        logger.log(
                            Level.WARNING,
                            "Job completion failed " + context.id(),
                            ee);
                    }
                }
            } catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                return;
            }

            if (context.length() <= 0) {
                yt.schedule((y) -> {
                    y.remove(task.processingPath());
                    return null;
                });
            } else {
                logger.warning("Task malformed");
                yt.schedule((y) -> {
                    y.move(task.processingPath(), task.malformedPath());
                    return null;
                });
            }

            logger.info("Completed " + context.toTskv().toString());
        } catch (YtException e) {
            logger.log(Level.WARNING, "Failed to process jobs", e);
            try {
                yt.schedule((yt) -> {
                    yt.move(task.processingPath(), task.todoPath());
                    return null;
                });
            } catch (InterruptedException ie) {
                logger.log(
                    Level.WARNING,
                    "Worker was interrupted",
                    ie);
            }
        } catch (InterruptedException ie) {
            logger.log(
                Level.WARNING,
                "Worker was interrupted " + task.toString(),
                ie);
        }
    }
    // CSON: ReturnCount
}
