package ru.yandex.search.mail.yt.consumer.upload;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.apache.http.message.BasicHttpRequest;

import ru.yandex.collection.BlockingBlockingQueue;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.mail.yt.consumer.YtClient;
import ru.yandex.search.mail.yt.consumer.YtConsumer;
import ru.yandex.search.mail.yt.consumer.YtException;
import ru.yandex.search.mail.yt.consumer.config.ImmutableWorkersConfig;
import ru.yandex.search.mail.yt.consumer.config.ImmutableYtConsumerConfig;

public class UploadWorkersManager
    implements Runnable, GenericAutoCloseable<IOException>
{
    public static final String READY_FOR_PROCESS = ".index";
    public static final String PROCESSING = ".processing";
    public static final String MALFORMED = ".malformed";
    public static final String COMPLETED = ".completed";

    private final YtClient ytClient;
    private final PrefixedLogger logger;
    private final ImmutableWorkersConfig workersConfig;
    private final ThreadPoolExecutor executor;

    private final String hostId;
    private final YtConsumer ytConsumer;

    private volatile boolean stop = false;

    private long lastCheckTime = 0;

    // CSOFF: ParameterNumber
    public UploadWorkersManager(
        final ImmutableYtConsumerConfig config,
        final YtConsumer ytConsumer)
    {
        this.ytConsumer = ytConsumer;
        this.ytClient = ytConsumer.yt();
        this.logger =
            ytConsumer.config().loggers().preparedLoggers().get(
                new RequestInfo(
                    new BasicHttpRequest(
                        RequestHandlerMapper.GET,
                        UploadWorker.WORKERS_LOGGER)));
        this.workersConfig = config.workersConfig();
        this.executor =
            new ThreadPoolExecutor(
                this.workersConfig.workers(),
                this.workersConfig.workers(),
                1,
                TimeUnit.MINUTES,
                new BlockingBlockingQueue<>(
                    new LifoWaitBlockingQueue<>(
                        workersConfig.workers())),
                new NamedThreadFactory("UploadWorker"));

        this.hostId = ytConsumer.getHostId();
    }
    // CSON: ParameterNumber

    public void start() {
        Thread thread = new Thread(this, "UploadWorkersManager");
        thread.start();
    }

    @SuppressWarnings("MixedMutabilityReturnType")
    private List<String> getNewTasks() throws InterruptedException {
        if (System.currentTimeMillis() - lastCheckTime
            < workersConfig.checkInterval())
        {
            return Collections.emptyList();
        }

        logger.info("Getting new task");
        List<String> result = new ArrayList<>();
        try {
            lastCheckTime = System.currentTimeMillis();
            for (String path: ytClient.list(workersConfig.todoPath())) {
                if (path.endsWith(READY_FOR_PROCESS)) {
                    result.add(
                        path.substring(
                            0,
                            path.length() - READY_FOR_PROCESS.length()));
                }
            }
        } catch (YtException e) {
            logger.log(Level.WARNING, "Failed to get new tasks", e);
        }

        logger.info("Found new jobs " + result.toString());
        return result;
    }

    public synchronized Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        status.put("queue", executor.getQueue().size());
        status.put("lastNewTasksChecked", lastCheckTime);
        return status;
    }

    private synchronized boolean scheduleJob(
        final String jobBaseName)
        throws InterruptedException
    {
        YtClient client;
        try {
            client = ytClient.startTransaction();
        } catch (YtException e) {
            logger.log(Level.WARNING, "Failed to start transaction", e);
            return false;
        }

        boolean status = false;
        try {
            UploadTask task =
                new UploadTask(jobBaseName, hostId, workersConfig);

            client.move(task.todoPath(), task.processingPath());
            client.commitTransaction();
            this.executor.execute(
                new UploadWorker(
                    task,
                    ytClient,
                    ytConsumer));

            status = true;
        } catch (YtException e) {
            logger.warning("Failed to schedule " + jobBaseName);
            try {
                client.abortTransaction();
            } catch (YtException se) {
                logger.log(Level.WARNING, "Failed to abort transaction", se);
            }
        }

        return status;
    }

    private boolean hasEmptySlots() {
        return this.executor.getQueue().size() < workersConfig.workers();
    }

    private void loadTasksFromDisk() throws YtException, InterruptedException {
        String postfix = '.' + hostId + PROCESSING;

        List<UploadTask> tasks = new ArrayList<>();
        for (String jobName : ytClient.list(workersConfig.processingPath())) {
            if (!jobName.endsWith(postfix)) {
                continue;
            }

            String baseName =
                jobName.substring(0, jobName.length() - postfix.length());

            tasks.add(new UploadTask(baseName, hostId, workersConfig));
        }

        for (UploadTask task: tasks) {
            logger.info("Task loaded " + task.toString());
            this.executor.execute(
                new UploadWorker(
                    task,
                    ytClient,
                    ytConsumer));
        }
    }

    private boolean initPaths() throws InterruptedException {
        try {
            if (!ytClient.exists(workersConfig.todoPath())) {
                logger.warning(
                    workersConfig.todoPath()
                        + " todo path not exists, creating");
                ytClient.createDirectory(workersConfig.todoPath(), true);
            }

            if (!ytClient.exists(workersConfig.donePath())) {
                logger.warning(
                    workersConfig.donePath()
                        + " done path not exists, creating");
                ytClient.createDirectory(workersConfig.donePath(), true);
            }

            if (!ytClient.exists(workersConfig.processingPath())) {
                logger.warning(
                    workersConfig.processingPath()
                        + " processing path not exists, creating");
                ytClient.createDirectory(
                    workersConfig.processingPath(),
                    true);
            }

            return true;
        } catch (YtException ye) {
            logger.log(Level.WARNING, "Failed to init pathes", ye);
            return false;
        }
    }

    @Override
    public void run() {
        logger.info("Starting workers manager, loading jobs");

        boolean loadJobsFromDisk = true;
        try {
            while (!stop) {
                logger.info("Trying to init pathes in yt");
                if (initPaths()) {
                    break;
                }

                Thread.sleep(workersConfig.checkInterval());
            }

            while (!stop) {
                if (loadJobsFromDisk) {
                    try {
                        loadTasksFromDisk();
                        loadJobsFromDisk = false;
                    } catch (YtException e) {
                        logger.log(
                            Level.WARNING,
                            "Failed to load jobs from disk",
                            e);
                        continue;
                    }
                }

                if (hasEmptySlots()) {
                    for (String job : getNewTasks()) {
                        synchronized (this) {
                            if (scheduleJob(job)) {
                                logger.info("Job scheduled " + job);
                            }

                            if (!hasEmptySlots()) {
                                break;
                            }
                        }
                    }
                }

                long sleepTime =
                    lastCheckTime
                        + workersConfig.checkInterval()
                        - System.currentTimeMillis();

                if (sleepTime > 0) {
                    Thread.sleep(sleepTime);
                }
            }
        } catch (InterruptedException e) {
            logger.log(Level.WARNING, "Worker manager interrupted", e);
            this.executor.shutdown();
        }
    }

    @Override
    public void close() throws IOException {
        this.executor.shutdown();
    }

    public void closeGracefully() throws IOException {
        this.executor.shutdown();
        stop = true;
    }
}
