package ru.yandex.dispatcher.common;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.Future;
import java.util.function.Supplier;

import org.apache.http.Header;
import org.apache.http.HeaderIterator;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.message.BasicHeader;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.collection.IntPair;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HeaderAsyncRequestProducerSupplier;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;

public abstract class AbstractHttpMessage implements HttpMessage {
    public static final Set<String> DROP_HEADERS =
        new TreeSet<>(String.CASE_INSENSITIVE_ORDER);

    static {
        DROP_HEADERS.add(HttpHeaders.ACCEPT);
        DROP_HEADERS.add(HttpHeaders.ACCEPT_CHARSET);
        DROP_HEADERS.add(HttpHeaders.ACCEPT_ENCODING);
        DROP_HEADERS.add(HttpHeaders.CONNECTION);
        DROP_HEADERS.add(HttpHeaders.CONTENT_LENGTH);
        DROP_HEADERS.add(HttpHeaders.EXPECT);
        DROP_HEADERS.add(HttpHeaders.HOST);
        DROP_HEADERS.add(HttpHeaders.REFERER);
        DROP_HEADERS.add(HttpHeaders.TRANSFER_ENCODING);
        DROP_HEADERS.add(HttpHeaders.USER_AGENT);
        DROP_HEADERS.add(YandexHeaders.ZOO_HTTP_METHOD);
    }

    protected List<Header> headers = null;
    protected String uri = null;
    protected boolean doWait = true;
    protected long createTime = 0L;
    protected long readTime = 0L;

    protected AbstractHttpMessage() {
    }

    protected AbstractHttpMessage(
        final String uri,
        final boolean doWait,
        final long createTime)
    {
        this.uri = uri;
        this.doWait = doWait;
        this.createTime = createTime;
        readTime = createTime;
    }

    protected abstract BasicAsyncRequestProducerGenerator request(
        final String relativeUri);

    protected abstract Supplier<HttpAsyncRequestProducer> request(
        final URI absoluteUri);

    private Supplier<HttpAsyncRequestProducer> createRequest(
        final HttpHost host,
        final String uri)
    {
        try {
            URI parsedUri = new URI(uri).parseServerAuthority();
            HttpHost parsedHost = URIUtils.extractHost(parsedUri);
            if (parsedHost != null) {
                return request(parsedUri);
            }
        } catch (URISyntaxException e) {
        }
        return () -> request(uri).apply(host);
    }

    @Override
    public Future<IntPair<Void>> run(
        final HttpHost host,
        final AsyncClient client,
        final long queueId,
        final int shard,
        final String service,
        final String cgiParams)
    {
        return runMessage(
            host,
            client,
            queueId,
            shard,
            service,
            cgiParams,
            EmptyFutureCallback.instance());
    }

    @Override
    public void run(
        final HttpHost host,
        final AsyncClient client,
        final long queueId,
        final int shard,
        final String service,
        final String cgiParams,
        final FutureCallback<IntPair<Void>> callback)
    {
        runMessage(
            host,
            client,
            queueId,
            shard,
            service,
            cgiParams,
            callback);
    }

    private Future<IntPair<Void>> runMessage(
        final HttpHost host,
        final AsyncClient client,
        final long queueId,
        final int shard,
        final String service,
        final String cgiParams,
        final FutureCallback<IntPair<Void>> callback)
    {
        String uri;
        if (cgiParams.isEmpty()) {
            uri = this.uri;
        } else if (this.uri.indexOf('?') == -1) {
            uri = this.uri + '?' + cgiParams;
        } else {
            uri = this.uri + '&' + cgiParams;
        }

        List<Header> headers = new ArrayList<>();
        headers.add(
            new BasicHeader(
                YandexHeaders.ZOO_QUEUE_ID,
                Long.toString(queueId)));
        headers.add(
            new BasicHeader(
                YandexHeaders.ZOO_SHARD_ID,
                Integer.toString(shard)));
        headers.add(new BasicHeader(YandexHeaders.ZOO_QUEUE, service));
        headers.add(
            new BasicHeader(
                YandexHeaders.ZOO_CTIME,
                Long.toString(createTime)));
        headers.add(
            new BasicHeader(
                YandexHeaders.ZOO_ATIME,
                Long.toString(readTime)));
        if (this.headers != null) {
            headers.addAll(this.headers);
        }

        return client.execute(
            new HeaderAsyncRequestProducerSupplier(
                createRequest(host, uri),
                headers),
            StatusCodeAsyncConsumerFactory.ANY_GOOD,
            callback);
    }

    @Override
    public String uri() {
        return uri;
    }

    @Override
    public void setHeader(final String name, final String value) {
        if (!DROP_HEADERS.contains(name)) {
            setHeader(new BasicHeader(name, value));
        }
    }

    private void setHeader(final Header header) {
        if (headers == null) {
            headers = new ArrayList<>();
        }
        headers.add(header);
    }

    @Override
    public void deleteHeader(final String name) {
        if (headers != null) {
            final Iterator<Header> iter = headers.iterator();
            while (iter.hasNext()) {
                if (iter.next().getName().equals(name)) {
                    iter.remove();
                }
            }
        }
    }

    @Override
    public void copyHeadersFrom(final HttpRequest request) {
        copyHeadersFrom(request, DROP_HEADERS);
    }

    public void copyHeadersFrom(
        final HttpRequest request,
        final Set<String> dropHeaders)
    {
        HeaderIterator iter = request.headerIterator();
        while (iter.hasNext()) {
            Header header = iter.nextHeader();
            if (!dropHeaders.contains(header.getName())) {
                setHeader(header);
            }
        }
    }

    @Override
    public boolean doWait() {
        return doWait;
    }

    @Override
    public long createTime() {
        return createTime;
    }

    @Override
    public long readTime() {
        return readTime;
    }

    protected void serializeHeaders(final DataOutputStream dout)
        throws IOException
    {
        if (headers != null) {
            dout.writeInt(headers.size());
            for (Header header: headers) {
                dout.writeUTF(header.getName());
                dout.writeUTF(header.getValue());
            }
        } else {
            dout.writeInt(0);
        }
    }

    protected void deserializeHeaders(final DataInputStream din)
        throws IOException
    {
        if (din.available() > 0) {
            final int count = din.readInt();
            for (int i = 0; i < count; i++) {
                setHeader(din.readUTF(), din.readUTF());
            }
        }
    }

    protected void dumpHeaders(final StringBuilder sb) {
        if (headers != null) {
            for (Header header: headers) {
                sb.append(header.getName());
                sb.append(':');
                sb.append('"');
                sb.append(header.getValue());
                sb.append('\r');
                sb.append('\n');
            }
        }
    }
}
