package ru.yandex.dispatcher.common;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.function.Supplier;

import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;

import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.util.timesource.TimeSource;

public class HttpPostMessage extends AbstractHttpMessage {
    private byte[] data;
    private ContentType contentType = null;

    public HttpPostMessage() {
    }

    public HttpPostMessage(
        final String uri,
        final byte[] data,
        final boolean doWait,
        final long createTime)
    {
        super(uri, doWait, createTime);
        this.data = data;
    }

    @Override
    protected BasicAsyncRequestProducerGenerator request(
        final String relativeUri)
    {
        return new BasicAsyncRequestProducerGenerator(
            relativeUri,
            data,
            contentType());
    }

    @Override
    protected Supplier<HttpAsyncRequestProducer> request(
        final URI absoluteUri)
    {
        return new AsyncPostURIRequestProducerSupplier(
            absoluteUri,
            data,
            contentType());
    }

    private ContentType contentType() {
        if (contentType == null) {
            return ContentType.APPLICATION_OCTET_STREAM;
        } else {
            return contentType;
        }
    }

    @Override
    public void setHeader(final String name, final String value) {
        if (name.equals(HttpHeaders.CONTENT_TYPE)) {
            contentType = ContentType.parse(value);
        } else {
            super.setHeader(name, value);
        }
    }

    @Override
    public void serialize(final DataOutputStream dout ) throws IOException {
        dout.writeUTF(uri);
        dout.writeInt(data == null ? 0 : data.length);
        if(data != null) {
            dout.write(data, 0, data.length);
        }
        dout.writeBoolean(doWait);
        dout.writeLong(createTime);
        serializeHeaders(dout);
    }

    @Override
    public void deserialize(final DataInputStream dis) throws IOException {
        uri = dis.readUTF();
        int dataLen = dis.readInt();
        data = new byte[dataLen];
        if (dataLen > 0) {
            dis.readFully(data, 0, dataLen);
        }
        doWait = dis.readBoolean();
        readTime = TimeSource.INSTANCE.currentTimeMillis();
        if (dis.available() > 0) {
            createTime = dis.readLong();
            deserializeHeaders(dis);
        }
    }

    @Override
    public byte[] toByteArray(
        final long queueId,
        final int shard,
        final String service)
    {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        StringBuilder sb = new StringBuilder();
        sb.append("POST ");
        sb.append(uri);
        sb.append(" HTTP/1.1\r\n");
        sb.append(YandexHeaders.ZOO_QUEUE_ID);
        sb.append(':');
        sb.append(' ');
        sb.append(queueId);
        sb.append('\r');
        sb.append('\n');

        sb.append(YandexHeaders.ZOO_SHARD_ID);
        sb.append(':');
        sb.append(' ');
        sb.append(shard);
        sb.append('\r');
        sb.append('\n');

        sb.append(YandexHeaders.ZOO_QUEUE);
        sb.append(':');
        sb.append(' ');
        sb.append(service);
        sb.append('\r');
        sb.append('\n');

        dumpHeaders(sb);
        if (contentType == null) {
            sb.append("Content-Type: application/json;\r\n");
        } else {
            sb.append("Content-Type: ");
            sb.append(contentType);
            sb.append('\r');
            sb.append('\n');
        }
        sb.append("Content-Length: ");
        sb.append(data.length);
        sb.append("\r\n\r\n");
        try {
            baos.write(sb.toString().getBytes(StandardCharsets.UTF_8));
            baos.write(data);
        } catch (IOException ign) {} //It should never happen for ByteArrayOutputStream
        return baos.toByteArray();
    }

    @Override
    public String toString() {
        return "HttpPostMessage[wait=" + doWait +
            ',' + data.length + " bytes of " + contentType() + ']';
    }
}

